function dmri_fiber_tck_file_worker_start(data_dir, worker_number)
% Process tck files.
%
% [Usage]
%    dmri_fiber_tck_file_worker_start(data_dir, worker_number);
%
% [Input]
%         data_dir : tck file directory.
%    worker_number : This is a number given by manager program
%
% [Output]
%    none
%
% [Note]
%    Many workers(MATLAB) concurrently launched to process tck files.
%    So, synchronous processing is required.
%    To realize it, directory is used as a exclusion access control.
%
%    1.Fiber tracking program creates directory.
%         data_dir/finished/tck_number
%    2.Worker program try to delete the directory.
%      if success, it means that the worker got a authority to 
%      process the tck file.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Worker create worker directory to notify a manager to ready to process.
% Note: The directory will be deleted by manager.
worker_dir = fullfile(data_dir, ['worker', num2str(worker_number)]);
mkdir(worker_dir);
log_file = fullfile(worker_dir, 'log.txt');
diary_flush(log_file);

try
parcel_mm_file = fullfile(data_dir, filesep, 'parcel_mm_coord.mat');

% track file 
finished_dir   = fullfile(data_dir, filesep, 'finished');
processed_dir  = fullfile(data_dir, filesep, 'processed');
% start processing
while(exist(worker_dir, 'dir') == 7)

    d = dir(finished_dir);

    for k=1:length(d)
        name = d(k).name;
        if strcmp(name, '.') || strcmp(name, '..')
            continue;
        end
        % To obtain an authority to process tck file, try to make a directory.
        % if the return value is success, the worker got an authority.
        [ret, warn] = mkdir(fullfile(processed_dir, filesep, name));
        % (mkdir always retuns 1.... second ret value check)

        % Get process authority
        if isempty(warn)
            % Success
            log_write(sprintf('%s.tck processing start\n', name));
            % Process tck file.
            tck_filename      = ['parcel', name, '.tck'];
            output_fiber_file = fullfile(data_dir, filesep, ['parcel_fiber', name, '.mat']);
            
            if exist(fullfile(data_dir, tck_filename), 'file') ~= 2
                error(sprintf('%s not found.', tck_filename));
            end
            save_connection(data_dir, tck_filename, parcel_mm_file, output_fiber_file);
            log_write(sprintf('%s.tck processing end\n', name));

            % delete processed name from tck list
            ret = rmdir(fullfile(finished_dir, filesep, name));
            diary_flush(log_file);
        else
            % Failure
            % log_write(sprintf('Failed to get a process authority %s.tck\n', name));
        end
        % Terminate
        if exist(worker_dir, 'dir') ~= 7, break; end
        pause(0.01);
    end
    pause(1);
end
catch
    errinfo = lasterror;
    log_write(errinfo.message);
    diary_flush;
    rethrow(errinfo);
end
diary_flush;
exit;

function log_write(message)
   fprintf('%s:%s', datestr(now, 31), message);
   
function diary_flush(diary_file)
    diary off;
    if exist('diary_file', 'var') && ~isempty(diary_file)
        diary(diary_file);
    end
 