function [connect, count, fiber_length, mfiber_length, Noverlap, track_parm] = ...
  calc_connection(dir_name, trackfile, ofile)
% Compute connection count and mean fiber length
% Input
%  dir_name: directory name where voxel images made by 'fsl2ascii' locate
%  trackfile: .tck file made by streamtrack in MRtrix [relative path to dir_name]
%  ofile: Output file containing mm coodinate of all the voxels in the mask
%  Nlabel: Number of label files
% Output
%  connect: Label of the tracks
%  count: Connection count for each label
%  fiber_length: Fiber length of the tracks
%  mfiber_length: mean fiber length for each label
%  Noverlap: Number of overlapped labels
%  track_parm: copy of 'tracks' other than 'tracks.data'

% 2012/06/15 M.Fukushima
% 2012/06/26 M.Fukushima
% 2014/05/29 rhayashi     Input argument Nlabel is removed.
%                         ofile contains Nlabel.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Initialize
Ntrack=0;
load(ofile, 'mcoord', 'label', 'Nlabel');
connect = cell(Ntrack,1);
count = zeros(Nlabel,1);
fiber_length = zeros(Ntrack,1);
mfiber_length = zeros(Nlabel,1);
Noverlap = zeros(Ntrack,1);
track_parm = [];

% Load files
filename = fullfile(dir_name, filesep, trackfile);
d = dir(filename);
if length(d)==1 && d.bytes > 0
    tracks = read_mrtrix_tracks(filename);
else
    return;
end

% Number of tracks
Ntrack = length(tracks.data);

% Extract coordinates of fiber endpoint
endcoord = zeros(Ntrack,3);
if Ntrack ~= 0
  for ii = 1:Ntrack
    if ~isempty(tracks.data{ii})
      endcoord(ii,:) = tracks.data{ii}(end,:);
    end
  end
end

% Shift mask coordinate (from the lowerleft corner to the center)
mcoord = [mcoord(:,1)-1 mcoord(:,2)+1 mcoord(:,3)+1];

% Step size
step_size = str2num(tracks.step_size);

% Initialize
connect = cell(Ntrack,1);
count = zeros(Nlabel,1);
fiber_length = zeros(Ntrack,1);
mfiber_length = zeros(Nlabel,1);
Noverlap = zeros(Ntrack,1);

if Ntrack ~= 0
  for ii = 1:Ntrack
    % sort distance between mask and fiber (endpoint) coordinate
    [inda, indb] = sort(sum((mcoord - repmat(endcoord(ii,:),[size(mcoord,1) 1])).^2,2));
    % Number of overlaps
    Noverlap(ii) = find(diff(inda)~=0,1);
    % Labeling the whole tracks
    connect{ii} = label(indb(1:Noverlap(ii)));
    % Count for each label
    count(connect{ii}) = count(connect{ii}) + 1/length(connect{ii});
    % Fiber length
    fiber_length(ii) = step_size * size(tracks.data{ii},1);
    % Sum of fiber length for each label
    mfiber_length(connect{ii}) ...
      = mfiber_length(connect{ii}) + fiber_length(ii)/length(connect{ii});
  end
  tmp = count;
  tmp(count==0) = 1;
  % Mean fibar length for each label
  mfiber_length = mfiber_length./tmp; clear tmp;
end

% Copy of 'tracks' other than 'tracks.data' 
track_parm = tracks;
track_parm.data = [];
