function dmri_label_file_to_FA_volume(label_dir, ...
                                      fs_dir, ref_struct_niigz_file, ...
                                      ref_FA_niigz_file, trans_info_dir, ...
                                      output_dir, Nprocess, mode)
% Create Parcel volume files on FA coordinate system from label 
% files(on FreeSurfer coordinate system).
%
% [Usage]
%    dmri_label_file_to_FA_volume(label_dir, ...
%                                 fs_dir, ref_struct_niigz_file, ...
%                                 re_FA_niigz_file, trans_info_dir, ...
%                                 output_dir, Npararel, mode)
% 
% [Input]
%                label_dir : The directory which contains parcel label files.
%                   fs_dir : Freesurfer directory.
%    ref_struct_niigz_file : reference structure image file (niftigz format).
%        ref_FA_niigz_file : reference FA(Fractional anisotrophy) image file (niftigz format).
%           trans_info_dir : The directory which cnotains transform infomation files
%                            created by dti_transwarp_info_create.m
%               output_dir : Parcels volume files will put into here.
%                 Nprocess : [Optional]The number of processes that 
%                                      want to use this conversion. Default = 4
%                     mode : [Optional] Processsing mode
%                              default  = 0 : Do all the operation(1-3)
%                                       = 1 : Parcel label to Parcel FA volume files.
%                                       = 2 : Parcel FA Volume files join to parcel_all.nii.gz
%                                             and parcel_mm_coord MAT-file.
%                                       = 3 : Create Parcel FA volumes exception files.
%
% [Output]
%    none
%
% [Output files]
%    N:Number of parcel files.
%
%    Volume files:
%        output_dir/parcel1.nii.gz to parcelN.nii.gz
%        output_dir/parcel_all.nii.gz
%        output_dir/parcel_ex1.nii.gz to parcel_exN.nii.gz
%
%     * parcel_ex1.nii.gz = pacel_all.nii.gz - parcel1.nii.gz
%
%    MAT-file:
%        output_dir/parcel_mm_coord.mat
%    Transform matrix file:
%        output_dir/qform % from FreeSurfer coord to FA mm coord.
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin < 6
    error('Please check input arguments.');
end
if ~exist('Nprocess', 'var')
    Nprocess = 2;
end
if exist(output_dir, 'dir') ~= 7
    mkdir(output_dir);
end
if ~exist('mode', 'var')
    mode = 0;
end

%
% --- Main Procedure
%

process_list   = cell(0);
start = tic;

% Define terminate function
function terminate_fcn
    % Kill fiber tracking process on specified hosts
    for k=1:length(process_list)
        process_list{k}.destroy();
    end
end

label_files    = dir([label_dir, filesep, 'parcel*.label']);
Nfiles         = length(label_files);
Nfile_per_proc = ceil(Nfiles / Nprocess);
work_dir       = tempname(output_dir);
mkdir(work_dir);

if mode == 0 || mode == 1
d = dir(fullfile(output_dir, filesep, 'parcel*.nii.gz'));
Nexist = length(d);
if Nexist ~= 0
    warning('Parcellation volume files are already exist. overwrite them.');
    delete(fullfile(output_dir, filesep, 'parcel*.nii.gz'));
    delete(fullfile(output_dir, filesep, 'parcel*.asc'));
    delete(fullfile(output_dir, filesep, 'qform'));
end

%%%%%%%%%%%%%%%%%%%%%%%%%%
% label to FA volume
%%%%%%%%%%%%%%%%%%%%%%%%%%
fprintf('Creating Parcel volume files from Parcel label files.\n');
files      = cell(0, 1);
fids       = cell(0, 1);
for k=1:Nfiles
    if mod(k, Nfile_per_proc) == 1
        Nth_script_file = length(files)+1;
        script_file     = fullfile(work_dir, filesep, ['script', num2str(Nth_script_file), '.sh']);

        % create template script
        isok = dmri_script_file_create(script_file);
        if isok == false, error('Failed to create script.'); end

        % open append mode
        fid = fopen(script_file, 'a');
        if fid == -1
            error('File open failed. processing terminated.');
        end
        fids{Nth_script_file}  = fid;
        files{Nth_script_file} = script_file;
    end

    % write command
    
    % label -> fs volume
    fs_vol_file     = fullfile(work_dir, filesep, ['tmp1_' num2str(k), '.nii.gz']);
    fprintf(fid,   ['mri_label2vol', ...
                   ' --label ' fullfile(label_dir, filesep, ['parcel' num2str(k), '.label']), ...
                   ' --temp ' fs_dir, '/mri/brain.nii.gz', ...
                   ' --o ' fs_vol_file ' --identity >/dev/null 2>&1 \n']);
    % fs volume -> struct volume
    struct_vol_file = fullfile(work_dir, filesep, ['tmp0_' num2str(k), '.nii.gz']);
    cmd = dmri_image_transform_cmd_get(fs_vol_file, struct_vol_file, ref_struct_niigz_file, 'fs2struct', trans_info_dir);
    fprintf(fid, '%s\n', cmd);
    
    % struct volume -> FA volume
    fa_vol_file     = fullfile(work_dir, filesep, ['tmp2_' num2str(k), '.nii.gz']);
    cmd = dmri_image_transform_cmd_get(struct_vol_file, fa_vol_file, ref_FA_niigz_file, 'struct2fa_nointerp', trans_info_dir);
    fprintf(fid, '%s\n', cmd);
    
    % Binarize volume
    fa_bin_file     = fullfile(output_dir, filesep, ['parcel' num2str(k), '.nii.gz']);
    fprintf(fid, '%s\n', ['fslmaths', ...
                         ' ' fa_vol_file,  ...
                         ' -thr 0', ...
                         ' -bin ', fa_bin_file]);

    % Convert Volume to ascii
    fa_ascii_file   = fullfile(output_dir, filesep, ['parcel' num2str(k), '-v.asc']);
    fprintf(fid, '%s\n', ['fsl2ascii', ...
                          ' ', fa_bin_file, ...
                          ' ', fa_ascii_file]);
    
    % remove '00000' at the end of filename
    fprintf(fid, '%s\n', ['mv', ...
                          ' ', [fa_ascii_file, '00000'], ...
                          ' ', fa_ascii_file]);
                          
end

% close script files
for k=1:length(fids), fclose(fids{k}); end


% Register clean up function(onCleanup is MATLAB builtin function)
onCleanup(@terminate_fcn);

% script files execution
runtime = java.lang.Runtime.getRuntime();
for k=1:length(files)
    cmd = files{k};
    process_list{length(process_list)+1} = runtime.exec(cmd);
end

% polling output directory
while(1)
    d = dir(fullfile(output_dir, filesep, 'parcel*.nii.gz'));
    Ncreated = length(d);
    fprintf('Now processing (%d/%d)\n', Ncreated, Nfiles);
    if Ncreated >= Nfiles
        disp('Done.');
        break;
    end
    pause(30);
end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% All Parcel volumes in one volume
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if mode == 0 || mode == 2
fprintf('All the Parcel volumes into single volume....');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Create qform
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
qform_file = fullfile(output_dir, filesep, 'qform');
cmd = ['fslorient -getqform ', fullfile(output_dir, filesep, 'parcel1.nii.gz'),...
      ' > ' qform_file];
dmri_system(cmd);

parcel_all_file = fullfile(output_dir, filesep, 'parcel_all.nii.gz');
copyfile(fullfile(output_dir, filesep, 'parcel1.nii.gz'), parcel_all_file);
for k=2:Nfiles
    % write command
    parcel_n_file = fullfile(output_dir, filesep, ['parcel' num2str(k), '.nii.gz']);
    dmri_add_images(parcel_all_file, parcel_n_file, parcel_all_file);
end
fprintf('Done.\n');
end

% Create parcel vertices in mm on FA coordinate.
if mode == 0 || mode == 2
disp('Converting parcels into FA coordinate vertices...');
parcels_mm_file = fullfile(output_dir, filesep, 'parcel_mm_coord.mat');
save_mask_coordinate(output_dir, qform_file, parcels_mm_file, ...
                     Nfiles, ref_FA_niigz_file);
delete(fullfile(output_dir, filesep, 'parcel*.asc'));

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Create ParcelN Ex mask files 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if mode == 0 || mode == 3
fprintf('Creating Parcel ex-mask volume files.\n');
parcel_all_file = fullfile(output_dir, filesep, 'parcel_all.nii.gz');
fids_ex_mask  = cell(0);
files_ex_mask = cell(0);
for k=1:Nfiles
    parcel_n_file    = fullfile(output_dir, filesep, ['parcel' num2str(k), '.nii.gz']);
    parcel_n_ex_file = fullfile(output_dir, filesep, ['parcel_ex' num2str(k), '.nii.gz']);
    if mod(k, Nfile_per_proc) == 1
        Nth_script_file = length(files_ex_mask)+1;
        script_file     = fullfile(work_dir, filesep, ['ex_mask_script', num2str(Nth_script_file), '.sh']);

        % create template script
        isok = dmri_script_file_create(script_file);
        if isok == false, error('Failed to create script.'); end

        % open append mode
        fid = fopen(script_file, 'a');
        if fid == -1
            error('File open failed. processing terminated.');
        end
        fids_ex_mask{Nth_script_file}  = fid;
        files_ex_mask{Nth_script_file} = script_file;
    end
    
    cmd = ['fslmaths ' parcel_all_file, ...
           ' -sub ',    parcel_n_file,   ...
           ' ',         parcel_n_ex_file];
    fprintf(fid, '%s\n', cmd);
end

% close script files
for k=1:length(fids_ex_mask), fclose(fids_ex_mask{k}); end

process_list = cell(0);

% Register clean up function(onCleanup is MATLAB builtin function)
onCleanup(@terminate_fcn);

% script files execution
for k=1:length(files_ex_mask)
    process_list{k} = runtime.exec(files_ex_mask{k});
end

% polling output directory
while(1)
    d = dir(fullfile(output_dir, filesep, 'parcel_ex*.nii.gz'));
    Ncreated = length(d);
    fprintf('Now processing (%d/%d)\n', Ncreated, Nfiles);
    if Ncreated >= Nfiles
        disp('Done.');
        break;
    end
    pause(30);
end
end

% delete working directory.
rmdir(work_dir, 's')

toc(start);

end
