function	vb_atlas_label(brainfile, atlas_id, Rmax)
% Create an area indices based on 
%   the Automated anatomical labeling 'aal'
%   or 'Brodmann' atlas file
% 
% ---- Syntax
% vb_atlas_label(brainfile,atlas_id,Rmax)
%
% ---- Input (Field names in 'parm')
% brainfile  : brain file path
% ---- Optional
% atlas_id   : ID of atlas ['aal']
%                 : 'aal' or 'brodmann'
% Rmax: Box size (radius of sphere) (mm)  [ 7 ]
%
%  Label is asigned to the label with maximum counting number 
%  inside the box (sphere) around the vertex by counting number of each label
%
% 2015/12/8 Made by M.Sato 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% ----- DEBUG
% parm=[];
% braindir  = 'D:\MatData\MNI-ICBM152\vbmeg\';
% brainfile = [braindir 'mni_icbm152_t1_tal_nlin_asym_09c.brain.mat'];
%% ----- DEBUG


if nargin < 2,	atlas_id = 'aal'; end;

% Box size for neighbor: 7 mm
if nargin < 3,	Rmax = 7; end;

%
% ---- Define atlas
%
atlas_dir  = 'atlas2vb_dir/MNI_atlas_templates/';
atlas_text = which([atlas_dir atlas_id '.txt']);
atlas_file = which([atlas_dir atlas_id '.img']);

% value corresponding to Cerebrum cortex region
%  AAL: 1-90: Cerebrum, 91-108: Cerebellum, 109-116: Vermis
%       Odd  : Left  &  Even : Right
% Brodmann: 1-47; brodmann area , 100- subcortical area
MaxCortex = 90;

EXT_brain = '.brain.mat';
brain_id  = brainfile(1:findstr(brainfile,EXT_brain)-1);

save_areafile  = [brain_id '_' atlas_id '.area.mat'];
save_atlasfile = [brain_id '_' atlas_id '.act.mat'];

% ----- Load MNI coordinate [m] of brain
Vmni = vb_load_cortex(brainfile, 'MNI')*1000;
Vmni = floor(Vmni);
Npoint = size(Vmni,1);
[Ndipole, NdipoleL] = vb_load_cortex_info(brainfile);

if Npoint ~= Ndipole,
	error('Brain file info error');
end

Vlabel = zeros(Npoint,1);
Vcount = zeros(Npoint,1);

% Cortex area index of 'V'
% Vinfo.cortexL = 
% Vinfo.cortexR = 
fprintf('Load Cortex area index\n')

load(brainfile, 'Vinfo');

Vindx  = [Vinfo.cortexL; Vinfo.cortexR];
Ncortex = length(Vindx);

%
% ---- Load Atlas template file
%      Template file is neurological format (right handed spm).
%

% ----- Read Template image-header
[dim, Trans] = vb_get_image_info(atlas_file);
invTrans = inv(Trans');

%[dim, Trans, XYZmm] = vb_get_image_info(atlas_file);
% dim(1:3) - the x, y and z dimensions of the volume
% Trans    - a 4x4 affine transformation matrix mapping from
%            voxel coordinates to MNI-mm-coordinates (RAS)
% XYZmm    = Trans * [XYZvox ; ones(1,Npoint)] [ 3 x Npoint]
%          : mm coordinate in MNI coordinate
%
%   [x  y  z  1] = [i  j  k  1] * Trans'
% XYZmni = Vox  * Trans'
% Vox    = XYZmni * inv(Trans')

% ----- Read atlas-template image 
% Zlabel - 3D image data (label value)
avw = avw_img_read(atlas_file);
Zlabel = avw.img;

Zlabel = Zlabel(:);
Nlabel = max(Zlabel);

% Make box index
x_indx = -Rmax:Rmax;
[X,Y,Z] = ndgrid(x_indx,x_indx,x_indx);
Box_ix = [X(:),Y(:),Z(:)];

% Select inside of sphere
ix = find( sum(Box_ix.^2,2) <= Rmax^2);
Box_ix = Box_ix(ix,:);

Nbox = size(Box_ix,1);
ilabel = (1:Nbox)';

for n = 1:Ncortex
	% MNI coordinate of np-th point
	np = Vindx(n);
	V  = Vmni(np,:);
	
	% Box coordinate around V
	XYZ = vb_repadd(Box_ix, V);
	
	% ----- Transform MNI-mm-coordinates to voxel coordinates
	Vox = fix( [XYZ ones(Nbox,1)] * invTrans );
	
	ix = find(  (Vox(:,1) > 0) ...
			  & (Vox(:,2) > 0) ...
			  & (Vox(:,3) > 0) ...
			  & (Vox(:,1) <= dim(1) ) ...
			  & (Vox(:,2) <= dim(2) ) ...
			  & (Vox(:,3) <= dim(3) ) ...
			  );
	
	% ----- Transform to 1D-index from [X,Y,Z] voxel coordinates
	Vindex = sub2ind(dim,Vox(ix,1),Vox(ix,2),Vox(ix,3));
	
	% Map label to vertex in the brain
	vlabel = Zlabel(Vindex) ;
	
	% if vlabel > MaxCortex = 90, it is in subcortical area
	jj = find( vlabel > MaxCortex );
	vlabel(jj) = 0;
	
	if strcmp( atlas_id, 'aal')
		if np > NdipoleL, 
		%  Right :  label = Even 
			jj = find( mod(vlabel,2) == 1);
			vlabel(jj) = 0;
		else
		%  Left : label = Odd 
			jj = find( mod(vlabel,2) == 0);
			vlabel(jj) = 0;
		end
	end
	
	% Make vlabel >= 1 by adding 1
	vlabel = vlabel + 1;
	
	% ---- Make SPARSE table to count label number in the box
	% S = SPARSE(i,j,s,m,n) 
	S = sparse(ix , vlabel, 1, Nbox, Nlabel+1);
	% S(i, j ) = 1 for j = vlabel(i)
	% count_label(j) : counting number of j-th label inside the box
	count_label = full(sum(S,1));
	
	% Find Max counting number (remove label=0 component)
	[count,imax] = max(count_label(2:Nlabel+1));
	
	if count > 0
		Vlabel(np) = imax(1);
		Vcount(np) = count;
		
		if strcmp( atlas_id, 'brodmann')
			if np > NdipoleL, 
				% Add 1000 to label in Right area
				Vlabel(np) = Vlabel(np) + 1000;
			end
		end
	end
end

%
% ---- Save labeled area
%

% Read atlas text file and extract area keys.
[label, label_name] = vb_read_atlas_label(atlas_text);

% Add Left-Right to label_name for 'brodmann'
if strcmp( atlas_id, 'brodmann')
	% Add 1000 to label in Right hemisphere
	label = [label(:) ; label(:) + 1000];
	
	% Add Left-Right to label_name
	N = length(label_name);
	for n = 1:N
		%name = strrep(label_name{n},'brodmann area ','brodmann_');
		name = label_name{n};
		label_name{n}   = [name '_L'];
		label_name{n+N} = [name '_R'];
	end	
	
end

%% Add Corpus label
%Narea = length(label);
%label(Narea+1) = corpus_label;
%label_name{Narea+1} = corpus_key;

% Intensity of 'Act.xxP' is the label for each vertex
Atlas.key = [atlas_id];
Atlas.xxP = Vlabel;
Atlas.label      = label;
Atlas.label_name = label_name;

if exist(save_atlasfile,'file')
	delete(save_atlasfile);
end
if exist(save_areafile,'file')
	delete(save_areafile);
end

% Save label as actfile
fprintf('Save the atlas file as "%s"  \n', save_atlasfile);
vb_add_act(save_atlasfile,Atlas,[],false);

% Save area label into area file
fprintf('Save the area file as "%s" \n', save_areafile);

vb_save_atlas_label(save_areafile,Vlabel,label,label_name);

NLABEL = sum( Vlabel > 0 );	% # of labeled points
fprintf('# of all vertex     = %d\n',Npoint)
fprintf('# of cortex points  = %d\n',Ncortex)
fprintf('# of labeled points = %d\n',NLABEL)

