function vb_plot_sensor_2d(x,z,clim, face_disp, highlight_channel_ix, XY_scale)
% Plot M/EEG spatial pattern on 2 dimentional surface.
% 
% [input]
% x                    : <<matrix>> Sensor position.
% z                    : <<vector>> Signal value.
% clim                 : <optional> <<vector>> Minimum and maximum of contour map.
% face_disp            : <optional> plot nose and ears(true or false) (default : false)
% highlight_channel_ix : <optional> <<vector>> highlight channel index(x's indices)
% XY_scale             : <optional> face radius expand/shrink ratio(default : 1.0)
%
% [note]
% If the coordinate system of the sensor position x is RAS (default of
% VBMEG), the contour plot will be drawn in the neurological convention,
% i.e., the view is above the subject and the right of the plot
% corresponds to the right of the subject. 
%
% [history]
% 2010-10-04 Taku Yoshioka
% 2012-06-20 taku-y
%  [trivial] Comment was modified. 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- Input arguments
if nargin<3, clim = []; end
if size(z,2)==1, z = z'; end
if ~exist('highlight_channel_ix', 'var')
    highlight_channel_ix = [];
end
if ~exist('XY_scale', 'var') || isempty(XY_scale)
    XY_scale = 1;
end

% --- Draw contour map
cla;

[phi,theta,r] = cart2sph(x(:,1),x(:,2),x(:,3));
[xa,ya,za] = pol2cart(phi,0.5*pi-theta,r);
% xa = mean(xa)-xa;
% ya = mean(ya)-ya;
ya = ya * -1;
xa = xa * -1;

dx = (max(xa)-min(xa))/200;
dy = (max(ya)-min(ya))/200;
r_max = max(sqrt(xa.^2+ya.^2));
xaa = xa;
yaa = ya;


xa = [xa; (XY_scale*1.1*r_max*cos(0:0.02*pi:1.99*pi))'];
ya = [ya; (XY_scale*1.1*r_max*sin(0:0.02*pi:1.99*pi))'];
za = [z'; zeros(100,1)];

% Colormap
warning('off', 'MATLAB:griddata:DuplicateDataPoints');
[X,Y,Z] = griddata(xa,ya,za,...
		   (min(xa):dx:max(xa)),...
		   (min(ya):dy:max(ya))','cubic');
warning('on', 'MATLAB:griddata:DuplicateDataPoints');
[C,h] = contourf(X,Y,Z,20);

if ~isempty(clim),
  caxis(clim);
else
  clim = get(gca,'CLim');
  if abs(clim(1))>abs(clim(2)), clim(2) = abs(clim(1));
  else clim(1) = -1*abs(clim(2)); end
  caxis(clim); 
end
set(h,'EdgeColor','none');
hold on;

% Contour line
z_max = max(abs(clim));
v = (0.2*z_max):(0.4*z_max):(1.0*z_max);
v = [-1*v(end:-1:1) v];
[C,h] = contourf(X,Y,Z,v);

% Rendering
if vb_matlab_version >= 7
  set(h,'Fill','off');
else
  set(h,'FaceColor','none');
end

plot(xaa,yaa,'ko','MarkerSize',2); 
set(gca,'XDir','reverse');
set(gca,'YDir','reverse');

if ~isempty(highlight_channel_ix)
    xaw = xa(highlight_channel_ix);
    yaw = ya(highlight_channel_ix);
    plot(xaw,yaw,'ko','MarkerSize',6, 'MarkerEdgeColor','w', 'MarkerFaceColor',[1, 1, 1]); 
end

if exist('face_disp', 'var') && face_disp
    vb_plot_sensor_2d_head_plot_add(gca);
end



return;