function	h = vb_plot_cross_section(V,F,Z0,vdim,dmax,Msize,Mtype,mode)
% plot cross section of surface on the slice
%   vb_plot_cross_section(V,F,Z0)
%   vb_plot_cross_section(V,F,Z0,vdim,dmax,Msize,Mtype,mode)
% --- Input
% V  : voxcel coordinate of the surface [Nvertex x 3]
% F  : patch index of the surface       [Npatch  x 3]
% Z0 : slice index for plot             [1 x 1]
% --- Output
% h  : plotted line handles
% --- Optional Input
% vdim  : slice cut direction
%       = 'x' : Sagittal cut : Y-Z plane
%       = 'y' : Coronal cut : X-Z plane
%       = 'z' : Transverse (Axial) cut : X-Y plane
% dmax  : search distance from cut slice (= 5)
% Msize : Size of plot line    (= 1)
% Mtype : Color of plot line   (= 'y')
% mode  : 2D plot mode for X-Y (= 0)
%       = 0   : plot without transpose
%       = 1   : plot by transposing 2D-image matrix
% --- Related function
%   vb_plot_vertex(V, vdim, Z0, dmax, Msize, Mtype, mode)
%
% --- Intersection point at Z = 0
% Line : V1 = [X1 Y1 Z1] to V2 = [X2 Y2 Z2]
%      : V  = (V1 - V2) * t + V2 
% z = 0 : 
% x = (X1*Z2 - X2*Z1)/(Z2 - Z1)
% y = (Y1*Z2 - Y2*Z1)/(Z2 - Z1)
%
% 2006-10-20 M. Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('vdim','var'),  vdim = 'z'; end;
if ~exist('dmax','var'),  dmax  = 5; end;
if ~exist('Mtype','var'), Mtype = 'y-'; end;
if ~exist('Msize','var'), Msize = 1; end;
if ~exist('mode','var'),  mode  = 0; end;

h = [];

switch	vdim
case	'x',
	% 'SAG' : Sagittal cut : Y-Z plane
	ax = 2;
	ay = 3;
	az = 1;
case	'y',
	% 'COR' : Coronal cut : X-Z plane
	ax = 1;
	ay = 3;
	az = 2;
case	'z',
	% 'TRN' : Transverse (Axial) cut : X-Y plane
	ax = 1;
	ay = 2;
	az = 3;
end

switch	mode
case	0,
	X  = V(:,ax);
	Y  = V(:,ay);
case	1,
	X  = V(:,ay);
	Y  = V(:,ax);
end

% slice axis coordinate 
Z = V(:,az) - Z0;

% Flag indicate upper & lower side of slice within dmax
Uix = (Z >= 0 & Z <= dmax);	% flag of upper side of slice
Lix = (Z <  0 & Z >= -dmax);% flag of lower side of slice

% Flag for patch index
FUix = Uix(F); 
FLix = Lix(F); 

% Patch include both upper and lower side of slice within dmax
Flist = find(sum(FUix,2) > 0 & sum(FLix,2) > 0);
Nlist = length(Flist);

for j=1:Nlist
	n  = Flist(j);	% patch id
	id = F(n,:);	% vertex index of patch
	up = FUix(n,:);	% Upper side flag of each vertex in the n-th patch
	dn = FLix(n,:);	% Lower side flag of each vertex in the n-th patch
	x  = [];		% x-coordinate to intersect the slice Z = 0
	y  = [];		% y-coordinate to intersect the slice Z = 0
	
	% Check two vertex of edge are in opposite side
	% : Check edge intersect the slice Z = 0
	%   and add intersection points to the list
	if (up(1)*dn(2) + up(2)*dn(1)) > 0,
		% Two vertex points of edge
		X1 = X(id(1));	X2 = X(id(2));
		Y1 = Y(id(1));	Y2 = Y(id(2));
		Z1 = Z(id(1));	Z2 = Z(id(2));
		
		if Z1 == Z2,
			% line is on the Z = 0 slice
			x = [x; X1; X2];
			y = [y; Y1; Y2];
		else
			% Intersection point at Z = 0
			x = [x; (X1*Z2 - X2*Z1)/(Z2 - Z1)];
			y = [y; (Y1*Z2 - Y2*Z1)/(Z2 - Z1)];
		end
	end
	
	if (up(2)*dn(3) + up(3)*dn(2)) > 0,
		% Two vertex points of edge
		X1 = X(id(2));	X2 = X(id(3));
		Y1 = Y(id(2));	Y2 = Y(id(3));
		Z1 = Z(id(2));	Z2 = Z(id(3));
		
		if Z1 == Z2,
			% line is on the Z = 0 slice
			x = [x; X1; X2];
			y = [y; Y1; Y2];
		else
			% Intersection point at Z = 0
			x = [x; (X1*Z2 - X2*Z1)/(Z2 - Z1)];
			y = [y; (Y1*Z2 - Y2*Z1)/(Z2 - Z1)];
		end
	end
	
	if (up(3)*dn(1) + up(1)*dn(3)) > 0,
		% Two vertex points of edge
		X1 = X(id(3));	X2 = X(id(1));
		Y1 = Y(id(3));	Y2 = Y(id(1));
		Z1 = Z(id(3));	Z2 = Z(id(1));
		
		if Z1 == Z2,
			% line is on the Z = 0 slice
			x = [x; X1; X2];
			y = [y; Y1; Y2];
		else
			% Intersection point at Z = 0
			x = [x; (X1*Z2 - X2*Z1)/(Z2 - Z1)];
			y = [y; (Y1*Z2 - Y2*Z1)/(Z2 - Z1)];
		end
	end
	
	% Plot intersection line of triangle with slice
	h = [h ; plot(x,y,Mtype,'MarkerSize',Msize)];
	hold on
end
