function [basis, basis_parm] = ...
  vb_job_leadfield_meg_harmonics(proj_root, basis_parm)
% actual worker function for MEG SPHERICAL HARMINICS
% [usage]
%   [basis, basis_parm] = vb_job_leadfield_meg_harmonics(proj_root, basis_parm)
% [input]
%    proj_root : <required> project root directory
%   basis_parm : <required> <<struct>>
%              :  data set in which the specification of calculation is defined
% [output]
%   none
% [note]
%   <<prior conditions>>
%     1) proj_root is valid
%     2) basis_parm is commonly valid
% [history]
%   2007-01-26 (Sako) initial version
%   2010-01-29 (Sako) replaced vb_load_meg_info with vb_load_measurement_info
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
[proj_root, basis_parm] = inner_check_arguments(proj_root, basis_parm);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% pre-processing
[Basis_mode, pick, Qpick, Wsensor, V0, V, xx] = ...
  vb_lf_preprocess(proj_root, basis_parm);

% Load head file
head_file   = [proj_root filesep basis_parm.head_file];
load([head_file], 'Vhead', 'Fhead', 'XXhead');
	
% Center of Head
if basis_parm.DEBUG == ON,
	% Setting for DEBUG to check Leadfield calculation
	Hcenter = [0 0 0];
else
	Hcenter = mean(Vhead);
end
	
% Change center of coordinate
Vhead = [Vhead(:,1)-Hcenter(1), Vhead(:,2)-Hcenter(2), Vhead(:,3)-Hcenter(3)];
V     = [    V(:,1)-Hcenter(1),     V(:,2)-Hcenter(2),     V(:,3)-Hcenter(3)];
pick  = [ pick(:,1)-Hcenter(1),  pick(:,2)-Hcenter(2),  pick(:,3)-Hcenter(3)];

% Calculate current direction according to 'Basis_mode'
[V,xx] = vb_current_vector(V, xx, Basis_mode);

fprintf('# of vertices = %d\n',size(Vhead,1));
	
% load meg file
meg_file = [proj_root filesep basis_parm.meg_file];
MEGinfo = vb_load_measurement_info(meg_file);
	
if ~isfield(MEGinfo,'SPHinfo') | basis_parm.Recalc == ON,
	   
	% average radius from center of head
	Rmax = vb_average_radius(Hcenter, Vhead);

	% function order
	func_order = basis_parm.func_order;
		
	% Calculate spherical harmonics coefficient
	SPHinfo.harmo_coef = ...
			vb_spherical_harmo_coef(Vhead, XXhead, func_order, Rmax, pick, Qpick);

	% Save SPHinfo to MEGinfo
	SPHinfo.func_order = func_order;
	SPHinfo.radius     = Rmax;
  SPHinfo.head_file  = basis_parm.head_file;
	MEGinfo.SPHinfo    = SPHinfo;

	vb_save(meg_file, 'MEGinfo');
else
	% Saved result
	SPHinfo = MEGinfo.SPHinfo;
end
	
%  MEG Spherical harmonics expansion
fprintf('--- MEG Spherical Harmonics expansion ');
N    = SPHinfo.func_order;
Rmax = SPHinfo.radius;
A    = SPHinfo.harmo_coef;

BB = vb_spherical_harmo_magnetic_field(V, xx, N, Rmax, pick, Qpick, A);
	
basis = BB * Wsensor';

%
% --- MAIN PROCEDURE --------------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
function [proj_root, basis_parm] = inner_check_arguments(proj_root, basis_parm)
if ~isfield(basis_parm, 'func_order'),
	% Order of spherical harmonics function
	basis_parm.func_order = 35;
end

% if you have other check points, you can do them here ...

return;
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
