function [basis, basis_parm] = vb_job_leadfield_eeg_bem(proj_root, basis_parm)
% EEG potencial for multilayer model by BEM 
%  [basis, basis_parm] = vb_job_leadfield_eeg_bem(proj_root, basis_parm)
%  basis : EEG Lead field matrix   ( NP x NS)
%
% basis_parm.brain_file : brain file (.brain.mat)
% basis_parm.area_file  : area file  (.area.mat)
% basis_parm.meg_file   : MEG data file (.meg.mat)
% basis_parm.Basis_mode : number of independent curent direction
%       = 1 :  current vextor = xx(n,:)
%       = 2 :  current vextor = orthogonal vectors to V(n,:)
%       = 3 :  current vextor = xx(n,:) & orthogonal vectors to xx(n,:)
% basis_parm.bem_mode   : Leadfield calculation method
%   MEG
%      0  : MEG spherical model (Sarvas)
%      1  : MEG BEM 1 shell model
%      2  : MEG spherical harmonics expansion
%   EEG
%      3  : EEG spherical 3 shell model
%      4  : EEG BEM 3 shell model
%
% basis_parm.sigma         : Conductivity of each region
% basis_parm.head_file     : Head surface file (.head.mat)
% --- Optional Input
% basis_parm.normal_mode : 
%       = 0 : normal vector at the vertex (Defualt)
%       = 1 : average normal vector in the neighbor of BV original brain 
%
% basis_parm.area_key      : Area key to specify possible current region
% basis_parm.Recalc        : Recalculation force flag
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% ---
%   V   : dipole location(in meters)               NP x 3
%   xx  : dipole moment direction                  NP x 3
%  pick : EEG sensors(in meters) on the scalp      NS x 3
% Omega : solid angle matrix for boundary surfaces
% Vhead : vertex coordinate for boundary surfaces
% Fhead : patch index for boundary surfaces
% Sout  : normal vector for boundary surfaces
%
% Nvertex   = 各境界面の頂点インデックス
%           = [ start_id(1)     end_id(1)     ; 
%                         ...                 ;
%               start_id(Nsurf) end_id(Nsurf) ]
%           = [開始インデックス, 終了インデックス]
%             innermost to outermost 
%
% Npatch	= 各境界面の三角面インデックス 
%           = [開始インデックス, 終了インデックス]
% Sigma 	= 各領域の伝導率 
%           = [sigma(1), ..., sigma(Nsurf)]
%
% 2008-10-9 Masa-aki Sato
%  structure of program is changed (algorithm is not changed)

% pre-processing
[Basis_mode, pick, Qpick, Wsensor, V0, V, xx] = ...
  vb_lf_preprocess(proj_root, basis_parm);

% check parameters and set default values if not be set
basis_parm = vb_util_check_head_shell_info(proj_root, basis_parm);

[V,xx] = vb_current_vector(V, xx, Basis_mode);

if iscell(basis_parm.head_file)
	headfile = basis_parm.head_file;
	joinfile = [vb_join_cell_file_name(headfile) '.head.mat'];
	joinpath = fullfile(proj_root, joinfile);
	
	if ~exist(joinpath,'file')
		sigma=[];layer=[];mode = 3; % MAX mode for radius
		vb_head_join_files(headfile,proj_root,joinpath, sigma,layer,mode);
	end
	basis_parm.head_file = joinfile;
end

new_headfile = fullfile(proj_root, basis_parm.head_file);
omega_file   = [vb_get_basename(new_headfile,'.head.mat') '.omega.mat'];

fprintf('Load %s\n',new_headfile)
load(new_headfile);
% Vhead : (# of vertex) x 3
% Fhead : (# of patch) x 3
% Nvertex : (# of surface) x 2
% Npatch  : (# of surface) x 2
% Sigma   : 1 x (# of surface)

fprintf('--- BEM for EEG\n')
fprintf('# of vertices = %d\n',size(Vhead,1));

if ~exist('Nvertex', 'var'), Nvertex = [1 length(Vhead)];          end;
if ~exist('Npatch', 'var'),  Npatch  = [1 length(Fhead)];          end;

Nsurf = size(Nvertex,1);
fprintf('# of surfaces = %d\n',Nsurf);

Sigma = basis_parm.sigma;

BEM.sigma   = [Sigma(1:Nsurf) 0];
BEM.Nvertex = Nvertex;
BEM.Npatch  = Npatch;

if exist(omega_file,'file')
	fprintf('Load %s\n',omega_file)
	load(omega_file)
end

if  ~exist('Omega','var') || basis_parm.Recalc == ON,
	tic;
	fprintf('--- Solid angle calculation \n');
		
	% (Normal vectors (XXhead) are outwarded here)
	[Omega, Sout] = vb_solid_angle_grk(Vhead,Fhead,XXhead);

	vb_ptime(toc);
	vb_fsave(omega_file,'Omega', 'Sout');
	fprintf('Saved omega file: %s\n',omega_file);
end

tic;

% 多層モデル補正
fprintf('--- Multisurface correction\n');
[Omega] = vb_bem_matrix(Omega,BEM);

% 自己重み係数：基底関数を三角面上で積分した寄与を加える
fprintf('--- Auto solid angle calculation\n');
Omega = vb_solid_auto_grk(Omega,Fhead,Sout);

vb_ptime(toc);

fprintf('--- EEG potential (BEM) \n');
	
% Electric potential
basis = vb_bem_eeg( Omega, Sout, Vhead, Fhead, V, xx, pick, BEM );
basis = basis';

%%% END OF FILE %%%
