function [err_code, Basis_mode, pick, Qpick, Wsensor, V0, V, xx] = ...
  vb_basisparm_check(basis_parm, proj_root, abort_swt)
% check and read data from basis_parm
% [usage]
%   [err_code, Basis_mode, pick, Qpick, Wsensor, V0, V, xx] = ...
%     vb_basisparm_check(basis_parm, proj_root, abort_swt)
% [input]
%   basis_parm : <required> <<struct>> 
%    proj_root : <required> <<string>> path of read data file
%    abort_swt : <optional> <<boolean>>
%              :   true : if error occur, abort with error message
%              : [false]: if error occur, return error code
% [output]
%     err_code : error code
%              : each bit means as follows
%              :  0x00) no error
%              :  0x01) basis_parm is empty
%              :  0x02) proj_root is empty
%              :  0x04) basis_parm.brain_file is invalid
%              :  0x08) basis_parm.meg_file is invalid
%              :  0x10) basis_parm.normal_mode is invalid
%              :  0x20) basis_parm.Basis_mode is invalid
%              :  0x40) basis_parm.area_key or area_file is invalid
%              :  0x80) coordinate type of basis_parm.meg_file is improper
%   Basis_mode : leadfield calculation mode
%         pick : [Nch x 3] sensor coordinates
%        Qpick : [Nch x 3] vector of each sensor coordinates
%      Wsensor : on-off for double sensor of YOKOGAWA
%           V0 : center of spherical model
%            V : brain model
%           xx : vector of brain model
% [note]
%   if error occur, this program doesnot make abort, and return error code
% [history]
%   2008-04-01 (Sako) initial version
%   2008-10-15 Taku Yoshioka Support '.sensortype' parameter
%   2010-01-29 (Sako) replaced vb_load_meg_info with vb_load_measurement_info
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('basis_parm', 'var'), basis_parm = []; end
if ~exist('proj_root', 'var'), proj_root = ''; end
if ~exist('abort_swt', 'var'), abort_swt = []; end
err_code = 0;
[err_code, basis_parm, proj_root, ABORT] = ...
  inner_check_arguments(err_code, basis_parm, proj_root, abort_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% ----- CONSTANTS
EC_BRAINFILE  = 4;
EC_MEGFILE    = 8;
EC_NORMALMODE = 16;
EC_BASISMODE  = 32;
EC_AREA       = 64;
EC_COORDTYPE  = 128;

% ----- required fields of basis_parm
% ----- brain_file
if ~isfield(basis_parm, 'brain_file')
  if ABORT
    error('brain_file is a required field of basis_parm');
  end
  basis_parm.brain_file = '';
  err_code = bitor(err_code, EC_BRAINFILE);
end

% ----- meg_file
if ~isfield(basis_parm, 'meg_file')
  if ABORT
    error('meg_file is a required field of basis_parm');
  end
  basis_parm.meg_file = '';
  err_code = bitor(err_code, EC_MEGFILE);
end

% ----- normal_mode
if ~isfield(basis_parm, 'normal_mode')
  if ABORT
    error('normal_mode is a required field of basis_parm');
  end
  basis_parm.normal_mode = [];
  err_code = bitor(err_code, EC_NORMALMODE);
end

if ~isfield(basis_parm, 'Basis_mode')
  if ABORT
    error('Basis_mode is a required field of basis_parm');
  end
  basis_parm.Basis_mode = [];
  err_code = bitor(err_code, EC_BASISMODE);
end

% ----- optional fields of basis_parm
if ~isfield(basis_parm, 'area_file')
	basis_parm.area_file = '';
end

if ~isfield(basis_parm, 'area_key')
	basis_parm.area_key = '';
end

% ----- check as file
if ~isempty(basis_parm.brain_file)
  brain_file = fullfile(proj_root, basis_parm.brain_file);

  if exist(brain_file, 'file') ~= 2
    if ABORT
      error('cannot read brain_file : ''%s''', brain_file)
    end
    err_code = bitor(err_code, EC_BRAINFILE);
    brain_file = '';
  end
else
  brain_file = '';
end

if ~isempty(basis_parm.area_file)
    area_file = fullfile(proj_root, basis_parm.area_file);
else
    area_file = '';
end

if ~isempty(basis_parm.area_key) && isempty(basis_parm.area_file)
  if ABORT
    error('area_file is a required if area_key is specified');
  end
  err_code = bitor(err_code, EC_AREA);
end

if ~isempty(basis_parm.area_file) && exist(area_file, 'file') ~= 2
  if ABORT
    error('cannot read area_file : ''%s''', area_file);
  end
  err_code = bitor(err_code, EC_AREA);
  area_file = '';
end

pick = [];
Qpick = [];
Wsensor = [];
V0 = [];
V = [];
xx = [];

% ----- MEG_FILE
if ~isempty(basis_parm.meg_file)
  meg_file   = fullfile(proj_root, basis_parm.meg_file);
  if exist(meg_file, 'file') ~= 2
    if ABORT
      error('cannot read meg_file : ''%s''', meg_file);
    end
    err_code = bitor(err_code, EC_MEGFILE);
  else
    % Load sensor info
    [pick, Qpick, Wsensor, V0] = vb_load_sensor(meg_file);

    % ----- coordinate type of meg_file
    coordtype = vb_msrmnt_load_coordtype(meg_file);
    if ~inner_validate_coordtype(coordtype)
      if ABORT
        error('invalid coordtype');
      end
      err_code = bitor(err_code, EC_COORDTYPE);
    end
  end
end

% basis_parm.Vcenter has priority over Vcenter of megfile
if isfield(basis_parm, 'Vcenter') && ~isempty(basis_parm.Vcenter)
  V0 = basis_parm.Vcenter;
end

Basis_mode  = basis_parm.Basis_mode;
area_key    = basis_parm.area_key;
normal_mode = basis_parm.normal_mode;

if ~isempty(brain_file)
  % Load cortical vertex coordinate and normal vector
  [V, xx] = vb_load_cortex_area(brain_file,area_file,area_key,normal_mode,'subj');
end

% Select channel (2008-11-27 Taku Yoshioka)
if ~isfield(basis_parm,'sensortype'), 
  basis_parm.sensortype = [];
end
sensortype = basis_parm.sensortype;
if ~isempty(sensortype), 
  MEGinfo = vb_load_measurement_info(meg_file);
  ix_sensor = find(MEGinfo.ChannelInfo.Type==sensortype);
  ix_sensor2 = [ix_sensor; ix_sensor+length(ix_sensor)];
  pick = pick(ix_sensor2,:);
  Qpick = Qpick(ix_sensor2,:);
  Wsensor = Wsensor(ix_sensor,ix_sensor2);
end

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [err_code, basis_parm, proj_root, abort_swt] = ...
  inner_check_arguments(err_code, basis_parm, proj_root, abort_swt)

if isempty(abort_swt), abort_swt = false; end

if abort_swt
  func_ = mfilename;
  if isempty(basis_parm)
    error('(%s)basis_parm is a required parameter', func_);
  end

else
  if isempty(basis_parm),  err_code = bitor(err_code, 1); end
end
return;
%
% --- end of inner_check_arguments()

% --- inner_validate_coordtype()
%
function result = inner_validate_coordtype(coordtype)
APPROPRIATE_COORD_TYPE = {'SPM_RIGHT_M'}; % ignore the case
coordtype = upper(coordtype);
%
% --- empty means O.K.
%
if isempty(coordtype)
  result = true;
  return;
end

type_len = length(APPROPRIATE_COORD_TYPE);
for i_type = 1:type_len
  if strcmp(APPROPRIATE_COORD_TYPE(i_type), coordtype)
    result = true;
    return;
  end
end
result = false;
return;
%
% --- end of inner_validate_coordtype()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
