function vb_headinfo_join(info_list, joined_file, verbose_swt)
% join plural Headinfo to single head file serially-concatenated
% [usage]
%   vb_headinfo_join(info_list, joined_file, verbose_swt)
% [input]
%     info_list : <required> array of plural Headinfo <<struct>>
%               :  [N x 1] or [1 x N]
%
%   joined_file : <optional> stored file path which has joined data
%               :  ['./joined_head.head.mat']
%
%   verbose_swt : <optional> switch to output verbose messages
%               :  [true] or false
% [output]
%   void
%
% [note]
%   stored data are as follow:
%     ((multi-layer head-mat file))
%        XXhead : [Npoint x 3]
%         Vhead : [NPoint x 3]
%         Fhead : [NPatch x 3] 
%       Nvertex : [NL x 2]  : start and end index
%        Npatch : [NL x 2]  : start and end index
%      Layer%ag : {NL} names of each layer
%
%   <<prior conditions>>
%     Each Headinfo has the complete fields set and is the same struct
%
%   @see vb_headinfo_make.m
%   @see vb_headinfo_save.m
%
% [history]
%   2007-04-10 (Sako) initial version
%   2007-06-15 (Sako) removed some fields from multi-layer head-mat file
%   2010-10-25 (Sako) modified how to get length of vhead
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

try
  % --- CHECK ARGUMENTS --- %
  if ~exist('info_list', 'var'),   info_list = []; end;
  if ~exist('joined_file', 'var'), joined_file = []; end;
  if ~exist('verbose_swt', 'var'), verbose_swt = []; end;
  
  [info_list, joined_file, VERBOSE] = ...
    inner_check_arguments(info_list, joined_file, verbose_swt);

  % --- check info_list
  if size(info_list,2) <= 1
    vb_headinfo_save(info_list(1), joined_file);
    return;
  end

  % --- MAIN PROCEDURE ------------------------------------------------------- %
  %

  % Rmethod
  %  --- All of the 'Rmethod' is expected to be the same
  Rmethod = vb_headinfo_get_rmethod(info_list(1));

  % get calculation mode by Rmethod
  if isempty(Rmethod)
    calc_mode = 1;
  else
    calc_mode = vb_util_convert_rmethod2calcmode(Rmethod);
  end
  
  % sort by average radius
  RR = [];
  N_info = size(info_list, 2);

  XXhead = [];
  Vhead = [];
  Fhead = [];
  Nvertex = [];
  Npatch = [];
  R = [];
  Sigma = zeros(1, N_info);
  LayerTag = cell(1, N_info);

  for ni = 1:N_info
    Vhead = vb_headinfo_get_vhead(info_list(ni));
    RR = [RR vb_util_distance_of_points(Vhead, calc_mode)];
  end
  
  [R idx] = sort(RR);
  
  InfoList = info_list(idx);

  % re-initialize Vhead
  Vhead = [];
  
  begin_v = 1;
  begin_p = 1;
  Npoint  = 0;

  for ni = 1:N_info
    cur_info = InfoList(ni);
  
    len_v = size(vb_headinfo_get_vhead(cur_info),1);
    len_p = size(vb_headinfo_get_fhead(cur_info),1);

    end_v = begin_v + len_v -1;
    end_p = begin_p + len_p -1;
    Nvertex = [Nvertex; begin_v end_v];
    Npatch  = [Npatch;  begin_p end_p];
    begin_v = end_v + 1;
    begin_p = end_p + 1;
  
    XXhead = [XXhead; vb_headinfo_get_xxhead(cur_info)];
    Vhead  = [ Vhead; vb_headinfo_get_vhead(cur_info)];
    Fhead  = [ Fhead; vb_headinfo_get_fhead(cur_info) + Npoint];
    
%     Sigma(ni)    = vb_headinfo_get_sigma(cur_info);
    sigma = vb_headinfo_get_sigma(cur_info);
    if isempty(sigma)
      sigma = NaN;
    end
    Sigma(ni) = sigma;
    LayerTag{ni} = vb_headinfo_get_layertag(cur_info);
    
    Npoint = Npoint + len_v;
  end

  % make Headinfo
  Headinfo = vb_headinfo_make([], Vhead, Fhead, XXhead, LayerTag, Sigma, ...
    Rmethod, R, Npatch, Nvertex);

  % save Headinfo
  vb_headinfo_save(Headinfo, joined_file);

  if VERBOSE
    fprintf('--- saved as current joined file : %s\n', joined_file);
  end

catch
  rethrow(lasterror);
end

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [info_list, joined_file, VERBOSE] = ...
  inner_check_arguments(info_list, joined_file, verbose_swt)

% required parameters
if isempty(info_list)
  error('info_list is a required parameter');  
end

% optional parameters
if isempty(joined_file)
  joined_file = ['.' filesep 'joined_head.head.mat'];
end

if isempty(verbose_swt)
  VERBOSE = true;
else
  VERBOSE = verbose_swt;
end

% arrange arrays to [1 x N]
info_list = vb_util_arrange_list(info_list, 1);
%
% --- end of inner_check_arguments()

%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %
%%% END OF FILE %%%
