function [P,dP] = vb_legendre_grad(N,x)
% vb_legendre_grad - Evaluate the first N Legendre polynomials
%  [P,dP] = vb_legendre_grad(N,x)
%
%  Evaluates the first N Legendre polynomials and 
%  the first derivatives at the vector x.
%
%  Input:
%    N  (scalar)   : Polynomial order to evaluate
%    x  (M x 1)    : Input vector of values for evaluation
%
%  Outputs:
%   P  (N x M)   : Legendre Polynomial
%                = [ P1(x1) P1(x2)    ...    P1(xM);
%                    P2(x1) P2(x2)    ...    P2(xM);
%                      .
%                      .
%                    PN(x1) PN(x2)    ...    PN(xM)]
% 
%  dP = dP/dx  :  Derivative of P  (N x M) 
% ---------------------- --------- 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if size(x,1)~=1, x=x'; end;

M  = size(x,2);      % # of evaluation points

P  = zeros(N,M); 
dP = zeros(N,M);

P(1,:)	= x;              % initial of P
P(2,:)	= 1.5*x.^2-0.5;  
dP(1,:) = ones(1,M);      % initial derivative

% recursively compute P,dP
for n=3:N,
  P(n,:) = ( (2*n-1).*x.*P(n-1,:) - (n-1)*P(n-2,:) )/n;
end

for n=2:N
  dP(n,:) = x.*dP(n-1,:) + n.*P(n-1,:);     
end
