function vb_job_grandaverage(proj_root,avr_parm)
% Calculate intersubject average of estimated cortical current.
% (VBMEG public function)
%
% [syntax]
% vb_job_current_grandaverage(proj_root,avr_parm)
%
% [input]
% proj_root: <<string>> VBMEG project root directory. 
% avr_parm : <<struct>> Parameters for intersubject averaging. 
% --- fields of avr_parm
%  curr_file    : <<cell>> Cortical current files. 
%  brain_file   : <<cell>> Cortical surface model files. 
%  key          : <<cell>> ID of coregistration matrix for each subjects
%                 corresponding to brain files. In common, coregistration
%                 matrix from the subject to a template cortical model is
%                 specified. 
%  reduce       : <optional> <<double>> Reduction ratio of vertices of
%                 the cortical surface model on which grandaverage is
%                 taken. 
%  tbrain_file  : <optional> <<string>> Filename of cortical surface
%                 model on which grandaverage is taken. This file is used
%                 to reduce the number of vertices. 
%  curr_file_out: <<string>> Filename of cortical current file. 
% ---
%
% [output]
% Cortical current file (.curr.mat) on the template cortical surface
% model (curr_file_out).
%
% [example]
% >> avr_parm.curr_file = {'./subjects/0001/result/0001_LR.curr.mat', ...
%                          './subjects/0002/result/0002_LR.curr.mat', ...
%                          './subjects/0003/result/0003_LR.curr.mat', ...
%                          './subjects/0004/result/0004_LR.curr.mat'};
% >> avr_parm.brain_file ...
%     = {'./subjects/0001/brain/0001.brain.mat', ...
%        './subjects/0002/brain/0002.brain.mat', ...
%        './subjects/0003/brain/0003.brain.mat', ...
%        './subjects/0004/brain/0004.brain.mat'};
% >> avr_parm.key = {'fsaverage.sphere.reg','fsaverage.sphere.reg', ...
%                    'fsaverage.sphere.reg','fsaverage.sphere.reg'};
% >> avr_parm.tbrain_file = './subjects/fsaverage/fsaverage.brain.mat';
% >> avr_parm.curr_file_out ...
%     = './subjects/fsaverage/result/fsaverage_LR.curr.mat';
% >> vb_job_grandaverage(proj_root,avr_parm);
%
% [history]
% 2010-12-01 Taku Yoshioka
% 2011-01-27 taku-y
%  [minor] 'reduce' parameter supported. If reduce is not given or
%  reduce==1.0, this function give the same result with that of the
%  previous version. 
% 2011-11-15 taku-y
%  [minor] Bug fixed for insufficient number of current dipoles, due to
%  small number of Z-current dipoles not covering whole cortex. 
% 2012-02-08 taku-y
%  [minor] Bug fixed for insufficient number of current dipoles, due to
%  small number of Z-current dipoles not covering whole cortex (first
%  loop). 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)


%
% Input arguments
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_struct2vars(avr_parm,{'curr_file','brain_file','key','reduce', ...
                    'tbrain_file','curr_file_out'});

if isempty(reduce),
  reduce = 1.0;
end

%
% Load first file
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[Jinfo,J] = vb_load_current(curr_file{1},0,true);
C         = vb_load_cortex_pmat(brain_file{1},key{1});

% added by taku-y 2012-02-08
Jtmp = zeros(size(C,2),size(J,2));
Jtmp(Jinfo.ix_act_ex,:) = J;
J = Jtmp;

if reduce==1.0, 
  Javr = C*J;
else
  brain_file_tmp = tbrain_file;
  
  % Reduce vertex of the cortical surface model
  V = vb_load_cortex(brain_file_tmp);
  ix_act = vb_reduce_cortex(brain_file_tmp,1:size(V,1),reduce);
  I = length(ix_act);
  
  % Calculate mean distance between reduced vertices to determine
  % smoothing filter size
  R = 0.0;
  N = 0;
  [nextDD,nextIX] = vb_load_cortex_neighbour(brain_file_tmp);
  for i=1:length(ix_act)
    [tmp,ixx] = union(nextIX{ix_act(i)},ix_act);
    R = R+sum(nextDD{ix_act(i)}(ixx));
    N = N+length(ixx);
  end
  R = R/N;
  
  % Calculate smoothing filter. It will be used for displaying purpose,
  % not affect further analysis.
  [W,ix_act_ex] ...
      = vb_spatial_gauss_filter(brain_file_tmp,R,2*R,ix_act,1,-1,1);
  W = W./repmat(sum(W,2),[1 size(W,2)]);
  
  % Load average current and map onto the template cortical surface
  C    = vb_load_cortex_pmat(brain_file{1},key{1});
  I    = size(C,2);
  T    = size(J,2);
  Jtmp = zeros(I,T);
  
  Jtmp(Jinfo.ix_act_ex,:) = J;
  Jtmp                    = C*Jtmp; % current on the template brain
  Zavr                    = Jtmp(ix_act,:);
end

%
% Load subsequent files and calculate intersubject average
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for i=2:length(curr_file)
  [Jinfo,J] = vb_load_current(curr_file{i},0,true);
  C         = vb_load_cortex_pmat(brain_file{i},key{i});
  
  % added by taku-y 2011-11-15
  Jtmp = zeros(size(C,2),size(J,2));
  Jtmp(Jinfo.ix_act_ex,:) = J;
  J = Jtmp;
  
  if reduce==1.0
    Javr = Javr+C*J;
  else
    % Map current onto the template cortical surface
    I    = size(C,2);
    Jtmp = zeros(I,T);
    
    Jtmp(Jinfo.ix_act_ex,:) = J;
    Jtmp                    = C*Jtmp; % current on the template brain
    Zavr                    = Zavr+Jtmp(ix_act,:);
  end
end

%
% Variables to be saved
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Jinfo = vb_load_current(curr_file{1},0,true);
Jinfo.Ntrial = 1;

if reduce==1.0
  NJact = size(C,1);
  Jinfo.ix_act     = 1:NJact;
  Jinfo.ix_act_ex  = 1:NJact;
  Jinfo.NJact      = NJact;
  Jinfo.Wact       = speye(NJact);

  load(curr_file{1},'bayes_parm');
  Jinfo.patch_norm = bayes_parm.patch_norm;
  
  Zact = Javr./length(curr_file);
else
  Jinfo.ix_act    = ix_act;
  Jinfo.ix_act_ex = ix_act_ex;
  Jinfo.NJact     = length(ix_act_ex);
  Jinfo.Wact      = W;
  
  load(curr_file{1},'bayes_parm');
  Jinfo.patch_norm = bayes_parm.patch_norm;
  
  Zact = Zavr./length(curr_file);
end
  
%
% Save result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_fsave(curr_file_out,'Zact','Jinfo');

%
% Save project file
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
proj_file = get_project_filename;
if isempty(proj_file), 
  return;
end;

project_file_mgr('load', proj_file);
project_file_mgr('add', 'avr_parm', avr_parm);

return;
