function [varargout] = tool_launcher(key, varargin)
% tool_launcher(key, varargin)
%    Tool launcher
%    fig = tool_launcher('init', <id>)
%
%      create figure to input parameter to launch tool.
%
%        id : 1 % job_view_cortex
%             2 % job_select_area
%             3 % job_edit_area
%             4 % job_plot_meg
%             5 % job_plot_currentmap
%             6 % job_edit_act
%             7 % vb_trial_extractor
%             8 % vb_signal_processor
%             9 % vb_view_head_2d
% brain_tool_launcher.fig
%  --- job_view_cortex, job_select_area, job_edit_area??
% current_tool_launcher.fig
%  --- job_plot_currentmap??
% meg_tool_launcher.fig
%  --- job_plot_meg??
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

switch(key)
    case 'init'
        id = varargin{1};
        varargout{1} = create_figure(id);
    case 'exit'
        fig = get(0, 'CurrentFigure');
        if ishandle(fig)
            try
                [startup_dir, component_handles, app_type] = get_app_data(fig);
                cd(startup_dir);
            catch
                % do nothing
            end
            delete(fig);
        end
    case 'callback'
        hObj = varargin{1};
        callback(hObj);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% ABSTRUCT FUNCTIONS
%
function [fig] = create_figure(id)
% Abstruct Create figure function
% [IN]
%    id : 1 % job_view_cortex
%         2 % job_select_area
%         3 % job_edit_area
%         4 % job_plot_meg
%         5 % job_plot_currentmap
%         7 % vb_trial_extractor
%         8 % vb_signal_processor
%         9 % vb_view_head_2d
% [OUT]
%    fig : figure handle

    %
    % --- Make figure
    %
    fig = [];

    % mapping from id to application type
    app_type = get_type(id);

    switch(app_type)
        case JOB_VIEW_CORTEX
            fig = job_view_cortex_launcher_figure;
        case JOB_SELECT_AREA
            fig = job_select_area_launcher_figure;
        case JOB_EDIT_AREA
            fig = job_edit_area_launcher_figure;
        case JOB_PLOT_MEG
            fig = job_plot_meg_launcher_figure;
        case JOB_PLOT_CMAP
            fig = job_plot_cmap_launcher_figure;
        case JOB_EDIT_ACT
            fig = job_edit_act_launcher_figure;
        case VB_TRIAL_EXTRACTOR
            fig = vb_trial_extractor;
        case VB_SIGNAL_PROCESSOR
            fig = vb_signal_processor;
        case VIEW_HEAD_2D
            fig = view_head_2d_launcher_figure;
    end

    %
    % --- Set application data to Figure
    %
    if ishandle(fig)
        set_app_data(fig, pwd, guihandles(fig), app_type);
    end

% end of create_figure()

function callback(hObj)
% Abstruct Callback function
% [IN]
%    hObj : action component
%
    fig = gcf;
    [startup_dir, H, app_type] = get_app_data(fig);

    switch(app_type)
        case JOB_VIEW_CORTEX
            job_view_cortex_callback(H, hObj);
        case JOB_SELECT_AREA
            job_select_area_callback(H, hObj);
        case JOB_EDIT_AREA
            job_edit_area_callback(H, hObj);
        case JOB_PLOT_MEG
            job_plot_meg_callback(H, hObj);
        case JOB_PLOT_CMAP
            job_plot_cmap_callback(H, hObj);
        case JOB_EDIT_ACT
            job_edit_act_callback(H, hObj);
        case VIEW_HEAD_2D
            view_head_2d_callback(H, hObj);
    end
% end of callback()

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  CONCRETE FUNCTIONS(create_figure)
%

function [fig] = job_view_cortex_launcher_figure
    % open figure
    fig = openfig('brain_tool_launcher.fig');
    set(fig, 'Name', 'View cortical model');
    h = guihandles(fig);

    % markup must parameter.
    set(h.brain_file_text, 'String', ...
                  ['*' get(h.brain_file_text, 'String')]);
% end of job_view_cortex_launcher_figure()

function [fig] = job_select_area_launcher_figure
    % open figure
    fig = openfig('brain_tool_launcher.fig');
    set(fig, 'Name', 'Make area data');
    h = guihandles(fig);
    
    % mark up must parameter.
    set(h.brain_file_text, 'String', ...
                  ['*' get(h.brain_file_text, 'String')]);
    set(h.act_file_text, 'String', ...
                  ['*' get(h.act_file_text, 'String')]);
    set(h.area_file_text, 'String', ...
                  ['*' get(h.area_file_text, 'String')]);
% end of job_select_area_launcher_figure()

function [fig] = job_edit_area_launcher_figure
    % open figure
    fig = openfig('brain_tool_launcher.fig');
    set(fig, 'Name', 'Edit area data');
    h = guihandles(fig);

    % Hide no need components
    set(h.act_file_text, 'Visible', 'off');
    set(h.act_file_edit, 'Visible', 'off');
    set(h.act_file_push, 'Visible', 'off');

    % mark up must parameter.
    set(h.brain_file_text, 'String', ...
                  ['*' get(h.brain_file_text, 'String')]);
    set(h.area_file_text, 'String', ...
                  ['*' get(h.area_file_text, 'String')]);

% end of job_edit_area_launcher_figure()

function [fig] = job_plot_meg_launcher_figure
    % open figure
    fig = openfig('meg_tool_launcher.fig');
    set(fig, 'Name', 'View MEG/EEG data');
    h = guihandles(fig);

    % mark up must parameter.
    set(h.meg_file_text, 'String', ...
                  ['*' get(h.meg_file_text, 'String')]);
% end of job_plot_meg_launcher_figure()

function [fig] = view_head_2d_launcher_figure
    % open figure
    fig = openfig('view_head_2d.fig');
    set(fig, 'Name', 'View Head model');
    h = guihandles(fig);
    
    set(h.mri_file_text, 'String', ...
                 ['*' get(h.mri_file_text, 'String')]);
    set(h.head_file_text, 'String', ...
                 ['*' get(h.head_file_text, 'String')]);

function [fig] = job_plot_cmap_launcher_figure
    % open figure
    fig = openfig('current_tool_launcher.fig');
    set(fig, 'Name', 'View estimated current');
    h = guihandles(fig);

    % mark up must parameter.
    set(h.current_file_text, 'String', ...
                  ['*' get(h.current_file_text, 'String')]);
    set(h.brain_file_text, 'String', ...
                  ['*' get(h.brain_file_text, 'String')]);

% end of job_plot_cmap_launcher_figure()

function [fig] = job_edit_act_launcher_figure
    % open figure
    fig = openfig('brain_tool_launcher.fig');
    set(fig, 'Name', 'Edit activity map');
    h = guihandles(fig);

    % Hide no need components
    set(h.area_file_text, 'Visible', 'off');
    set(h.area_file_edit, 'Visible', 'off');
    set(h.area_file_push, 'Visible', 'off');
    
    % Move component position(to area file)
    t_pos = get(h.area_file_text, 'Position');
    e_pos = get(h.area_file_edit, 'Position');
    p_pos = get(h.area_file_push, 'Position');
    
    set(h.act_file_text, 'Position', t_pos);
    set(h.act_file_edit, 'Position', e_pos);
    set(h.act_file_push, 'Position', p_pos);

    % mark up must parameter.
    set(h.brain_file_text, 'String', ...
                  ['*' get(h.brain_file_text, 'String')]);
    set(h.act_file_text, 'String', ...
                  ['*' get(h.act_file_text, 'String')]);

% end of job_edit_area_launcher_figure()

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  CONCRETE FUNTIONS(callback)
%

function job_view_cortex_callback(H, hObj)
% Receive job_view_cortex launcher application input
%
% [IN]
%       H : component handles
%    hObj : action object
%
    switch(hObj)
        case H.ok_push
            brain_file = get(H.brain_file_edit, 'String');
            area_file  = get(H.area_file_edit, 'String');
            act_file   = get(H.act_file_edit, 'String');
            if ~exist(brain_file, 'file')
                errordlg('Cortical model file is not specified.', 'error');
                return;
            end
            close; % This figure close
            vb_act_area_viewer(brain_file, act_file, area_file);
        case H.brain_file_push
            [absolute_file, cancelled] = file_select('.brain.mat', ...
                                                     'Select cortical model file');
            if ~cancelled
                set(H.brain_file_edit, 'String', absolute_file);
            end
        case H.area_file_push
            [absolute_file, cancelled] = file_select('.area.mat', ...
                                                     'Select cortical area file');
            if ~cancelled
                set(H.area_file_edit, 'String', absolute_file);
            end
        case H.act_file_push
            [absolute_file, cancelled] = file_select('.act.mat', ...
                                                     'Select cortical activity file');
            if ~cancelled
                set(H.act_file_edit, 'String', absolute_file);
            end
    end
% end of job_view_cortex_callback()

function job_select_area_callback(H, hObj)
% Receive job_select_area launcher application input
%
% [IN]
%       H : component handles
%    hObj : action object
%
    switch(hObj)
        case H.ok_push
            brain_file = get(H.brain_file_edit, 'String');
            area_file  = get(H.area_file_edit, 'String');
            act_file   = get(H.act_file_edit, 'String');
            if ~exist(brain_file, 'file')
                errordlg('Cortical model file is not specified.', 'error');
                return;
            elseif ~exist(area_file, 'file')
                errordlg('Cortical area file is not specified.', 'error');
                return;
            elseif ~exist(act_file, 'file')
                errordlg('Cortical activity file is not specified.', 'error');
                return;
            else
                close; % This figure close
                job_select_area(brain_file, area_file, act_file);
            end
        case H.brain_file_push
            [absolute_file, cancelled] = file_select('.brain.mat', ...
                                                     'Select cortical model file');
            if ~cancelled
                set(H.brain_file_edit, 'String', absolute_file);
            end
        case H.area_file_push
            [absolute_file, cancelled] = file_select('.area.mat', ...
                                                     'Select cortical area file');
            if ~cancelled
                set(H.area_file_edit, 'String', absolute_file);
            end
        case H.act_file_push
            [absolute_file, cancelled] = file_select('.act.mat', ...
                                                     'Select cortical activity file');
            if ~cancelled
                set(H.act_file_edit, 'String', absolute_file);
            end            
    end
% end of job_select_area_callback()


function job_edit_area_callback(H, hObj)
% Receive job_edit_area launcher application input
%
% [IN]
%       H : component handles
%    hObj : action object
%
    switch(hObj)
        case H.ok_push
            brain_file = get(H.brain_file_edit, 'String');
            area_file  = get(H.area_file_edit, 'String');
            if ~exist(brain_file, 'file')
                errordlg('Cortical model file is not specified.', 'error');
                return;
            elseif ~exist(area_file, 'file')
                errordlg('Cortical area file is not specified.', 'error');
                return;
            else
                close; % This figure close
                job_edit_area(brain_file, area_file);
            end
        case H.brain_file_push
            [absolute_file, cancelled] = file_select('.brain.mat', ...
                                                     'Select cortical model file');
            if ~cancelled
                set(H.brain_file_edit, 'String', absolute_file);
            end
        case H.area_file_push
            [absolute_file, cancelled] = file_select('.area.mat', ...
                                                     'Select cortical area file');
            if ~cancelled
                set(H.area_file_edit, 'String', absolute_file);
            end
    end
% end of job_edit_area_callback()

function view_head_2d_callback(H, hObj)
% Receive view_head_2d launcher application input
%
% [IN]
%       H : component handles
%    hObj : action object
%
    switch(hObj)
        case H.ok_push
            head_file = get(H.head_file_edit, 'String');
            mri_file  = get(H.mri_file_edit, 'String');
            if exist(head_file, 'file') ~= 2
                errordlg('HEAD file is not specified.', 'error');
                return;
            end
            if exist(mri_file, 'file') ~= 2
                errordlg('MRI file is not specified.', 'error');
                return;
            end
            close; % This figure close
            drawnow;
            vb_view_head_2d(mri_file, head_file);
        case H.mri_file_push
            [absolute_file, cancelled] = file_select({'.nii', '.hdr'}, ...
                                                     'Select MRI file');
            if ~cancelled
                set(H.mri_file_edit, 'String', absolute_file);
            end
        case H.head_file_push
            [absolute_file, cancelled] = file_select({'.head.mat'}, ...
                                                     'Select Head file');
            if ~cancelled
                set(H.head_file_edit, 'String', absolute_file);
            end
    end
% end of view_head_2d_callback()
            
function job_plot_meg_callback(H, hObj)
% Receive job_plot_meg launcher application input
%
% [IN]
%       H : component handles
%    hObj : action object
%
    switch(hObj)
        case H.ok_push
            meg_file = get(H.meg_file_edit, 'String');
            if ~exist(meg_file, 'file')
                errordlg('MEG(EEG)file is not specified.', 'error');
                return;
            else
                close; % This figure close
                job_plot_meg(meg_file);
            end
        case H.meg_file_push
            [absolute_file, cancelled] = file_select({'.meg.mat', '.eeg.mat'}, ...
                                                     'Select MEG/EEG data');
            if ~cancelled
                set(H.meg_file_edit, 'String', absolute_file);
            end
    end
% end of job_plot_meg_callback()

function job_plot_cmap_callback(H, hObj)
% Receive job_plot_cmap launcher application input
%
% [IN]
%       H : component handles
%    hObj : action object
%
    switch(hObj)
        case H.ok_push
            current_file  = get(H.current_file_edit, 'String');
            brain_file = get(H.brain_file_edit, 'String');
            if ~exist(current_file, 'file')
                errordlg('Current file is not specified.', 'error');
                return;
            elseif ~exist(brain_file, 'file')
                errordlg('Cortical model file is not specified.', 'error');
                return;
            else
                close; % This figure close
                job_plot_currentmap(current_file, brain_file);
            end
        case H.current_file_push
            [absolute_file, cancelled] = file_select('.curr.mat', ...
                                                     'Select current timecourse file');
            if ~cancelled
                set(H.current_file_edit, 'String', absolute_file);
            end
        case H.brain_file_push
            [absolute_file, cancelled] = file_select('.brain.mat', ...
                                                     'Select cortical model file');
            if ~cancelled
                set(H.brain_file_edit, 'String', absolute_file);
            end

    end
% end of job_plot_cmap_callback()

function job_edit_act_callback(H, hObj)
% Receive job_edit_area launcher application input
%
% [IN]
%       H : component handles
%    hObj : action object
%
    switch(hObj)
        case H.ok_push
            brain_file = get(H.brain_file_edit, 'String');
            act_file  = get(H.act_file_edit, 'String');
            if ~exist(brain_file, 'file')
                errordlg('Cortical model file is not specified.', 'error');
                return;
            elseif ~exist(act_file, 'file')
                errordlg('Cortical activity file is not specified.', 'error');
                return;
            else
                close; % This figure close
                job_edit_act(brain_file, act_file);
            end
        case H.brain_file_push
            [absolute_file, cancelled] = file_select('.brain.mat', ...
                                                     'Select cortical model file');
            if ~cancelled
                set(H.brain_file_edit, 'String', absolute_file);
            end
        case H.act_file_push
            [absolute_file, cancelled] = file_select('.act.mat', ...
                                                     'Select cortical act file');
            if ~cancelled
                set(H.act_file_edit, 'String', absolute_file);
            end
    end
% end of job_edit_act_callback()

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  APPLICATION DATA MANAGEMENT
%
function set_app_data(fig, startup_dir, components, app_type)
% This function saves application data to figure.
% [IN]
%            fig : figure handle
%    startup_dir : startup directory.
%                  This application change current directory.
%                  startup_dir is to rollback directory.
%     components : component handles including this figure.
%       app_type : application type

    data.pwd       = pwd;
    data.app_type  = app_type;
    data.H         = guihandles(fig);
    set(fig, 'UserData', data);

% end of set_app_data()

function [startup_dir, components, app_type] = get_app_data(fig)
% This function loads application data from figure.
% [IN]
%    fig : figure handle.
% [OUT]
%    startup_dir : startup directory.
%                  This application change current directory.
%                  startup_dir is to rollback directory.
%     components : component handles including this figure.
%       app_type : application type

    data = get(fig, 'UserData');
    startup_dir = data.pwd;
    components = data.H;
    app_type    = data.app_type;

% end of get_app_data()

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% UTILITY FOR THIS APPLICATION


function [absolute_file, cancelled] = file_select(extension, caption)
% This function gives FileSelection dialog.
% [IN]
%    extension : cell or character array.
%                e.g. '.meg.mat', {'.meg.mat', '.eeg.mat'}
%      caption : dialog title
% [OUT]
%    absolute_file : absolute file path
%        cancelled : true or false. true means file selection is cancelled.
%
    absolute_file = [];
    cancelled = false;
    
    % Open dialog
    h = file_dialog;
    if ischar(extension)
        extension = cellstr(extension);
    end
    h = set(h, 'file_extensions', extension);
    h = set(h, 'current_dir', pwd);
    if exist('caption', 'var')
        h = set(h, 'dialog_title', caption);
    end
        
    [dirname fnames] = visible(h);
    
    % Cancel pushed?
    if isempty(dirname)
        cancelled = true;
    else
        cd(dirname);
        absolute_file = fullfile(dirname, fnames{1});
    end

% end of file_select()

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DEFINITION SECTION
%

function [app_type] = get_type(id)
% This function do mapping between id and app_type.
% [IN]
%    id : application id
% [OUT]
%    app_type : application type

    switch(id)
        case 1
            app_type = JOB_VIEW_CORTEX;
        case 2
            app_type = JOB_SELECT_AREA;
        case 3
            app_type = JOB_EDIT_AREA;
        case 4
            app_type = JOB_PLOT_MEG;
        case 5
            app_type = JOB_PLOT_CMAP;
        case 6
            app_type = JOB_EDIT_ACT;
        case 7
            app_type = VB_TRIAL_EXTRACTOR;
        case 8
            app_type = VB_SIGNAL_PROCESSOR;
        case 9
            app_type = VIEW_HEAD_2D;
    end
% end of get_type()

function [def] = JOB_VIEW_CORTEX
    def = 1;
function [def] = JOB_SELECT_AREA
    def = 2;
function [def] = JOB_EDIT_AREA
    def = 3;
function [def] = JOB_PLOT_MEG
    def = 4;
function [def] = JOB_PLOT_CMAP
    def = 5;
function [def] = JOB_EDIT_ACT
    def = 6;
function [def] = VB_TRIAL_EXTRACTOR
    def = 7;
function [def] = VB_SIGNAL_PROCESSOR
    def = 8;
function [def] = VIEW_HEAD_2D
    def = 9; 


