function [trial_list] = vb_continuous_file_extract_trial(obj, trig_list, trial_setting)
% Extract trial list for each trigger.
% [USAGE]
%   trial_list = continous_file_extract_trial(obj, trig_list, trial_setting);
% [IN]
%              obj : continous_file object
%        trig_list : trigger parameter list {N}
%    trial_setting : trial_setting
%                          .pretrigger_ms  : pretrigger length of trials 
%                          .posttrigger_ms : posttrigger length of trials
%                     .minumum_distance_ms : minimum distance between trials
%                                            if empty is specified, distance won't be checked.
% [OUT]
%       trial_list : extracted trial list  {trials x N}
%
% [see also]
%    vb_job_trial_onset;
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin < 3
    error('This function needs 3 arguments. Please check your code.');
end
%
% --- Main Procedure
%
if ~isfield(trial_setting, 'minimum_distance_ms')
    trial_setting.minimum_distance_ms = [];
end
trial_list = cell(0);

freq = vb_continuous_file_get_sample_freq(obj);
minimum_distance = freq * trial_setting.minimum_distance_ms/1000;

for k=1:length(trig_list)

    %
    % --- create parameter
    %

    % trigger property
    parm = vb_trigger_get_parameter(trig_list{k});

    % data property
    parm.fsamp = vb_continuous_file_get_sample_freq(obj);

    % trial property
    parm.Pretrigger_ms  = trial_setting.pretrigger_ms;
    parm.Posttrigger_ms = trial_setting.posttrigger_ms;

    %
    % --- Trial extraction
    %

    % load status channel data
    load_spec = struct;
    load_spec.ChannelName = parm.status_ch;
    status_data = vb_load_meg_data(obj.filename, load_spec);
    vb_disp(sprintf('extracting from continuous_file :%s....', obj.filename'));

    [ix_trial, trig, cond_id, status_val, status] = ...
        vb_get_trial_time_index(status_data, parm);

    % create Trial set
    trial_set = [];
    Ntrial = size(ix_trial, 2);
    for j=1:Ntrial
        trial_samples = ix_trial(:, j);

        from  = trial_samples(1);
        to    = trial_samples(end);
        onset = trig(j);

        trial = vb_trial_new(from, to, onset);

        if ~isempty(trial_set)
            previous_onset = trial_set(end).onset;
            distance = onset - previous_onset;
            if distance < minimum_distance
                continue;
            end
        end
        trial_set = [trial_set; trial];        
    end

    % Register trial_set
    trial_list{k} = trial_set;
end

%
% --- After check
%
if nargout < 1
    error('your program didn''t receive extracted trial set list.');
end
