function vb_signal_processor_apply_processing(input_files, ...
                                              output_files, ...
                                              filter_list, ...
                                              mode)
% Apply filters to input_files and put output_files.
% [USAGE]
% vb_signal_processor_apply_processing(input_files, ...
%                                             output_files, ...
%                                             filter_list);
% [IN]
%    input_files  : input MEG/EEG files.    [cell string]
%    output_files : filtered MEG/EEG files. [cell string]
%     filter_list : filter list             [cell struct]
%            mode : display mode
%                   = 0 : progress is shown by command line. [default]
%                   = 1 : progress is shown by dialog.
% [OUT]
%    none
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('input_files', 'var') || ~iscellstr(input_files)
    error('input files must be cell string');
end
if ~exist('output_files', 'var') || ~iscellstr(output_files)
    error('output files must be cell string');
end
if ~exist('filter_list', 'var')
    error('filter_list is a required parameter.');
end
if ~exist('mode', 'var')
    mode = 0;
end
if ~(mode == 0 || mode == 1)
    error('Invalid mode was specified.');
end

%
% --- Main Procedure
%
Nfiles   = length(input_files);
Nfilters = length(filter_list);
d = vb_define_signal_processor;

vb_define_device;
for k=1:Nfiles
    str = sprintf('Now Processing... (%d/%d)', k, Nfiles);
    if mode == 0
        % progress by command line 
        vb_disp(str);
    elseif mode == 1
        % progress by dialog
        if exist('h', 'var') && ~isempty(h) && ishandle(h)
            delete(h);
        end
        h = msgbox(str, 'Please wait', 'modal');
        set(findobj(h, 'Tag', 'OKButton'), 'Visible', 'off');
        drawnow;
    end
    
    input_file  = input_files{k};
    output_file = output_files{k};

    % EEG Common reference
    [device, format, Nsample, org_freq, pretrig] = get_file_info(input_file);
    for f=1:length(filter_list)
        if strcmp(filter_list{f}.type, d.TYPE_PROCESSING_COMMON_REFERENCE)
            if strcmp(format, 'external')
                vb_disp('All the EEG data will be loaded at once to apply common reference filter even though the file format is external.');
                format = 'internal'; % load all at once;
            end
        end
    end
        
    %%%%%%%%%%%%%%%%%%%%%%%%%%%
    % external format section
    %%%%%%%%%%%%%%%%%%%%%%%%%%%
    if strcmp(format, 'external')
        info = vb_load_measurement_info(input_file);
        if strcmp(device, 'MEG')
            ext = FILE_EXT_BIN_CH_MEG;
            output_dir     = [strrep(output_file, '.meg.mat', '') '_bin'];
            save_precision = vb_meginfo_get_precision(info);
            [p, f] = vb_get_file_parts(output_dir);
            rel_output_dir = ['./', f];
            filter_ch  = vb_megfile_get_channel_label_meg(input_file);
            extra_ch   = [vb_megfile_get_channel_label_extra(input_file); ...
                          vb_megfile_get_channel_label_refmg(input_file)];
        else
            ext = FILE_EXT_BIN_CH_EEG;
            output_dir = [strrep(output_file, '.eeg.mat', '') '_bin'];
            save_precision = vb_eeginfo_get_datatype(info);
            [p, f] = vb_get_file_parts(output_dir);
            rel_output_dir = ['./', f];
            filter_ch = vb_eegfile_get_channel_label(input_file, [], 1);
            extra_ch  = vb_eegfile_get_channel_label(input_file, [], 2);
        end

        % Create precision settings
        Nch       = size(filter_ch, 1);
        Nch_extra = size(extra_ch,  1);
        if isempty(save_precision)
            save_precision = 'float64';
        end
        if size(save_precision, 1) == 1
            % MEG case, save_precision is only defined one.
            filt_ch_precision = cellstr(repmat(save_precision, Nch, 1));
            ext_ch_precision  = cellstr(repmat(save_precision, Nch_extra, 1));
        elseif iscellstr(save_precision)
            % EEG case, precision is defined for each channel.
            filt_ch_precision = save_precision(1:Nch);
            ext_ch_precision  = save_precision(Nch+1:end);
        end
            
        % Process channel data
        for ch=1:Nch
            % load single channel data
            data = load_ch_data(device, input_file, filter_ch{ch});
            data_freq = org_freq;
            
            % filtering
            only_down_sampling = false;
            for j=1:Nfilters
                [data, data_freq] = ...
                    apply_filter(data, data_freq, ...
                                 filter_list{j}, only_down_sampling);
            end
            % save ch data
            save_data_to_dir(output_dir, ext, data, filter_ch{ch}, filt_ch_precision{ch});
        end
 
        % Process extra channel data
        for ch=1:Nch_extra
            % load single channel data
            data = load_ch_data(device, input_file, extra_ch{ch});
            data_freq = org_freq;
            
            % filtering(downsampling only)
            only_down_sampling = true;
            for j=1:Nfilters
                [data, data_freq] = ...
                    apply_filter(data, data_freq, ...
                                 filter_list{j}, only_down_sampling);
            end
            % save ch data
            save_data_to_dir(output_dir, ext, data, extra_ch{ch}, ext_ch_precision{ch});
        end
        
        % Modifying header information
        new_info = struct;
        if org_freq ~= data_freq
            new_info.n_sample = size(data, 2);
            new_info.sampling_freq = data_freq;
            new_info.pretrigger    = round((data_freq/org_freq)*pretrig);
        end
        new_info.bin_dir       = rel_output_dir;
        switch(device)
            case 'MEG'
                % create MEG-MAT
                vb_saver_meg_copy_info(input_file, output_file, new_info);
            case 'EEG'
                % create EEG-MAT
                vb_saver_eeg_copy_info(input_file, output_file, new_info);
        end

    else
    %%%%%%%%%%%%%%%%%%%%%%%%%%%
    % internal format section
    %%%%%%%%%%%%%%%%%%%%%%%%%%%
        if strcmp(device, 'MEG')
            % MEG internal format
            new_data = struct;
            %
            % --- channel data
            %
            new_data.bexp = vb_load_meg_data(input_file);
            data_freq     = org_freq;
            if ~isempty(new_data.bexp)
                % filtering
                only_down_sampling = false;
                for j=1:Nfilters
                    [new_data.bexp, data_freq] = ...
                        apply_filter(new_data.bexp, data_freq, ...
                                     filter_list{j}, only_down_sampling);
                end
            end
            
            %
            % --- extra channel
            %
            load_spec = struct;
            load_spec.ChannelType = 'EXTRA';
            new_data.bexp_ext = vb_load_meg_data(input_file, load_spec);
            data_freq     = org_freq;
            if ~isempty(new_data.bexp_ext)
                % filtering(downsampling only)
                only_down_sampling = true;
                for j=1:Nfilters
                    [new_data.bexp_ext, data_freq] = ...
                        apply_filter(new_data.bexp_ext, data_freq, ...
                                     filter_list{j}, only_down_sampling);
                end
            end

            %
            % --- reference data
            %
            load_spec = struct;
            load_spec.ChannelType = 'REFERENCE';
            new_data.refmg = vb_load_meg_data(input_file, load_spec);
            data_freq  = org_freq;
            if ~isempty(new_data.refmg)
                % filtering(downsampling only)
                only_down_sampling = true;
                for j=1:Nfilters
                    [new_data.refmg, data_freq] = ...
                        apply_filter(new_data.refmg, data_freq, ...
                                     filter_list{j}, only_down_sampling);
                end
            end
            
            % Modifying header information
            new_info = struct;
            if org_freq ~= data_freq
                new_info.n_sample = size(new_data.bexp, 2);
                new_info.sampling_freq = data_freq;
                new_info.pretrigger    = round((data_freq/org_freq)*pretrig);
            end
            % create MEG-MAT
            vb_saver_meg_copy_info(input_file, output_file, new_info);
            
            % save MEG data
            vb_msrmnt_store_data(input_file, new_data, filter_list, output_file);
        else
            % EEG internal format

            %
            % --- channel data
            %
            data = vb_load_meg_data(input_file);
            data_freq     = org_freq;
            if ~isempty(data)
                % filtering
                only_down_sampling = false;
                for j=1:Nfilters
                    [data, data_freq] = ...
                        apply_filter(data, data_freq, ...
                                     filter_list{j}, only_down_sampling);
                end
            end
            %
            % --- extra channel
            %
            load_spec = struct;
            load_spec.ChannelType = 'EXTRA';
            ext_data = vb_load_meg_data(input_file, load_spec);
            data_freq     = org_freq;
            if ~isempty(ext_data)
                % filtering(downsampling only)
                only_down_sampling = true;
                for j=1:Nfilters
                    [ext_data, data_freq] = ...
                        apply_filter(ext_data, data_freq, ...
                                     filter_list{j}, only_down_sampling);
                end
            end
            % Modifying header information
            new_info = struct;
            if org_freq ~= data_freq
                new_info.n_sample = size(data, 2);
                new_info.sampling_freq = data_freq;
                new_info.pretrigger    = round((data_freq/org_freq)*pretrig);
            end
            % create EEG-MAT
            vb_saver_eeg_copy_info(input_file, output_file, new_info);
            
            % save EEG data
            new_data = struct;
            new_data.eeg_data = [data; ext_data];
            vb_msrmnt_store_data(input_file, new_data, filter_list, output_file);
        end
    end
end
if exist('h', 'var') && ~isempty(h) && ishandle(h)
    delete(h);
else
    vb_disp('done.');
end

function data = load_ch_data(device, device_file, ch_name)
% load data
% device      : 'MEG' or 'EEG'
% device_file : MEG-MAT or EEG-MAT
% ch_name     : char or cell string

if ischar(ch_name)
    ch_name = {ch_name};
end
load_spec.ChannelName = ch_name;
load_spec.ChannelSwitch = true; % Channek Name is to read
load_spec.ChannelType   = device;
load_spec.ActiveChannel = 0; % all channels

data = vb_load_meg_data(device_file, load_spec);

function save_data_to_dir(data_dir, ext, data, ch_name, precision)
% save data
% device      : 'MEG' or 'EEG'
% device_file : MEG-MAT or EEG-MAT
% ch_name     : char or cell string
if isempty(precision), precision = 'float64'; end

save_spec.ch_label = ch_name;
save_spec.data     = data;
save_spec.dir      = data_dir;
save_spec.precision= precision;
save_spec.ext      = ext;
vb_saver_cmn_make_binary_file(save_spec);

function [data, freq] = apply_filter(data, freq, filter, only_down_sampling)
% [Input]
%   data               : measurement data(ch x time x trial)
%   freq               : sampling frequency of the data.
%   filter             : filter parameter for vb_filter_data.
%   only_down_sampling : true or false [default:false]
% [Output]
%   data               : processed data.
%   freq               : sampling frequency of the processed data.
if ~exist('only_down_sampling', 'var')
    only_down_sampling = false;
end

d = vb_define_signal_processor;
switch(filter.type)
    case d.TYPE_PROCESSING_DOWNSAMPLE
        % fall through
    otherwise
        if only_down_sampling
            return;
        end
end

%
% --- preprocess data
%

% convert gui parameter to library parameter
lib_filter = vb_signal_processor_util_convert_parm_to_lib_parm(filter, freq);
[data, freq] = vb_filter_data(data, lib_filter);

function [device, format, Nsample, freq, pretrigger] = ...
    get_file_info(device_file)
% device     : 'MEG' or 'EEG'
% format     : 'internal' or 'external'
% Nsample    : Sampling number of the file.
% freq       : Sampling frequency of the file.
% pretrigger : pretrigger length of the file.

device = vb_load_device(device_file);
info   = vb_load_measurement_info(device_file);

Nsample    = vb_info_get_sample_number(info);
freq       = vb_info_get_sampling_frequency(info);
pretrigger = vb_info_get_pre_trigger(info);

switch(device)
    case 'MEG'
        [state, guide_def] = vb_util_check_variable_in_matfile(device_file, 'bexp');
    case 'EEG'
        [state, guide_def] = vb_util_check_variable_in_matfile(device_file, 'eeg_data');
    otherwise
        error('Unknown Device data');
end
if (state == guide_def.VALID)
    % innner format
    format = 'internal';
else
    format = 'external';
end
