function pa_head_util(fig, command)
% This function is subcontracting function of head GUI.
% [USAGE]
%    pa_head_util(<fig>, <command>);
% [IN]
%        fig : figure handle of head gui.
%    command : command
% [OUT]
%    none
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('fig', 'var') || isempty(fig) || ~ishandle(fig)
    error('invalid figure handle is specified.');
end
if ~exist('command', 'var') || isempty(command)
    error('invalid command is specified.');
end

%
% --- Main Procedure
%

% load data(@see pa_head_create)
data = guidata(fig);

switch(command)
    case 'update_exec_push_status'
        update_exec_push_status(data.H);
    case 'update_output_fname'
        update_output_fname(data.H);
    case 'update_model_type'
        update_model_type(data.H);
    case 'update_csf_source'
        update_csf_source(data.H);
    case 'update_skull_source'
        update_skull_source(data.H);
    case 'update_scalp_source'
        update_scalp_source(data.H);
    case 'reset_parameter'
        reset_parameter(data.H);
    case 'exec_head'
        exec_head(data);
    case {'view2d', 'view3d'}
        view_model(data.H, command);
end

% save data
guidata(fig, data);

function update_output_fname(H)
% This function makes output filename and sets filename to GUI.
% [IN]
%    H : GUI component handles
% [OUT]
%    none

    global vbmeg_inst;
    define = vbmeg_inst.const;

    head_file = get(H.head_file_edit, 'String');
    [f_path, f_name, f_ext] = vb_get_file_parts(head_file);
    head_file = [f_name, f_ext];

    % save_dir
    save_dir = get(H.save_dir_edit, 'String');

    % head filename(If it is empty, create from Analyze filename.)
    if isempty(head_file)
        analyze_file = get(H.analyze_file_edit, 'String');
        [f_path, f_name, f_ext] = vb_get_file_parts(analyze_file);
        if isempty(f_name)
            return;
        else
            head_file = [f_name, '.head.mat'];
        end
    end

    %
    % --- make filename
    %

    head_file = fullfile(save_dir, head_file);
    set(H.head_file_edit, 'String', head_file);

    update_exec_push_status(H);

function update_exec_push_status(H)
% This function changes Exec button status.(clickable or not)
% [IN]
%    H : GUI component handles
% [OUT]
%    none

    head_parm = pa_head_get_parm(H.figure);
    [ok, msg] = pa_head_check_parm(H.figure, head_parm);
    if ok
        set(H.exec_push, 'Enable', 'on');
    else
        set(H.exec_push, 'Enable', 'off');
    end

function exec_head(data)
% This function execute job_head with set parameter.
% [IN]
%    data : data of head gui(@see pa_head_create)
% [OUT]
%    none

    H = data.H;

    % get parameter from gui
    head_parm = pa_head_get_parm(H.figure);

    % parameter check
    [ok, msg] = pa_head_check_parm(H.figure, head_parm);
    if ~ok
        errordlg(msg);
        return;
    end

    %
    % --- execute job
    %
    set(H.exec_push, 'String', 'Processing...');
    proj_root = get(H.prjroot_edit, 'String');
    vb_job_head_3shell(proj_root, head_parm);
    set(H.exec_push, 'String', 'Exec');
    
function reset_parameter(H)
% This function resets head GUI.
% [IN]
%    H : GUI component handles
% [OUT]
%    none

return;


function update_model_type(H)
% This function update screen by model type
% [IN]
%    H : GUI component handles
% [OUT]
%    none

    update_vertices(H);
    update_skull_source(H);
    update_scalp_source(H);

function update_vertices(H);
% This function update vertices number
% [IN]
%    H : GUI component handles
% [OUT]
%    none
    vertices_component = [
        H.vertices_1shell_edit
        H.vertices_3shell_edit
    ];
    set(vertices_component, 'Visible', 'off');

    if get(H.model_3shell_radiobutton, 'Value')
        set(H.vertices_3shell_edit, 'Visible', 'on');
    else
        set(H.vertices_1shell_edit, 'Visible', 'on');
    end

function update_csf_source(H)
% This function update screen by csf source
% [IN]
%    H : GUI component handles
% [OUT]
%    none

    fs_component = [
        H.fs_csf_file_text
        H.fs_csf_file_edit
        H.fs_csf_file_push
        H.fs_morphology_text
        H.fs_morphology_edit];

    spm_component = [
        H.spm_gray_matter_file_text
        H.gray_file_edit
        H.gray_file_push
        H.brain_file_text
        H.brain_file_edit
        H.brain_file_push
        H.gray_morphology_text
        H.gray_morphology_edit
        fs_component];

    curry_component = [
        H.curry_file_text
        H.curry_file_edit
        H.curry_file_push
        H.curry_morphology_text
        H.curry_morphology_edit];

    all_component = [spm_component; fs_component; curry_component];
    set(all_component, 'Visible', 'off');

    source_type = get(H.csf_source_popup, 'Value');
    switch(source_type)
        case 1
        % SPM segmentation file + Cortical model file
            set(spm_component, 'Visible', 'on');
        case 2
        % Curry v4.5 surface file
            set(curry_component, 'Visible', 'on');
        otherwise
            error('unknown csf source was specified.');
    end
    
function update_skull_source(H)
% This function update skull panel status
% [IN]
%    H : GUI component handles
% [OUT]
%    none
    panel_component = [
         H.skull_source_text
         H.skull_source_frame
         H.skull_source_type_text
         H.skull_source_popup];
    fs_component = [
         H.fs_skull_file_text
         H.fs_skull_file_edit
         H.fs_skull_file_push];
    scalp_component = [
         H.scalp_thickness_text
         H.scalp_thickness_edit
         H.scalp_thickness_unit_text];
    skull_thickness_component = [
         H.min_skull_thickness_text
         H.min_skull_thickness_edit
         H.min_skull_thickness_unit_text];

    all_component = [panel_component; fs_component;
                     scalp_component; skull_thickness_component];
    set(all_component, 'Visible', 'off');
%    if get(H.model_3shell_radiobutton, 'Value')
        set(panel_component, 'Visible', 'on');
        if get(H.skull_source_popup, 'Value') == 1
            % Use Scalp thickness
            set(scalp_component, 'Visible', 'on');
            set(skull_thickness_component, 'Visible', 'on');
        else
            % FreeSurfer surface file(outer skull)
            set(fs_component, 'Visible', 'on');
            set(skull_thickness_component, 'Visible', 'on');
        end
 %   end

function update_scalp_source(H)
% This function update scalp panel status
% [IN]
%    H : GUI component handles
% [OUT]
%    none
    panel_component = [
        H.scalp_source_text
        H.scalp_source_frame
        H.scalp_source_type_text
        H.scalp_source_popup];
    fs_component = [
        H.fs_scalp_file_text
        H.fs_scalp_file_edit
        H.fs_scalp_file_push];
    face_component = [
        H.face_file_text
        H.face_file_edit
        H.face_file_push];

    all_component = [panel_component; fs_component; face_component];
    set(all_component, 'Visible', 'off');

%    if get(H.model_3shell_radiobutton, 'Value')
        set(panel_component, 'Visible', 'on');
        if get(H.scalp_source_popup, 'Value') == 1
            % FreeSurfer surface file(outer skin)
            set(fs_component, 'Visible', 'on');
        else
            % Face surface file
            set(face_component, 'Visible', 'on');
        end
%    end
    
    
function view_model(H, command)

    analyze_file = get(H.analyze_file_edit, 'String');
    if exist(analyze_file, 'file') == 0
        errordlg('Analyze file should be specified.');
        return;
    end

    switch(command)
        case 'view2d'
            % Set 2D view parameter
            view2d_setting = bm_view_parm2d_new;
            view2d_setting = bm_view_parm2d_set_analyze_file(view2d_setting, analyze_file);

            view2d = bm_edit_app_view2d_new;
            view2d = bm_edit_app_view_update(view2d, H.figure, ...
                                             @pa_head_load_display_factor, ...
                                             view2d_setting);
            
        case 'view3d'
            % Set 3D view parameter
            view3d_setting = bm_view_parm3d_new;
            view3d = bm_edit_app_view3d_new;
            view3d = bm_edit_app_view_update(view3d, H.figure, ...
                                             @pa_head_load_display_factor, ...
                                             view3d_setting);
    end
    return;
