function H = pa_fmri_util(fig, command, parm)
% This function is subcontracting function of fMRI GUI.
% [USAGE]
%    pa_brain_util(<fig>, <command>);
% [IN]
%        fig : figure handle of brain gui.
%    command : command
%       parm : parameter
% [OUT]
%    none
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('fig', 'var') || isempty(fig) || ~ishandle(fig)
    error('invalid figure is specified.');
end
if ~exist('command', 'var') || isempty(command)
    error('command is a required parameter.');
end

%
% --- Main Procedure
%

% load data(@see pa_fmri_create)
data = guidata(fig);

switch(command)
    case 'update_exec_push_status'
        update_exec_push_status(data.H);
    case 'set_spm_result_file'
        spm_result_file = parm;
        set_spm_result_file(data, spm_result_file);
    case 'set_spm_type'
        mode = parm;
        set_spm_type(data.H, mode);
    case 'exec_fmri'
        exec_fmri(data);
end

function set_spm_result_file(data, spm_result_file)
% This function fmri activity keys from SPM result file(SPM.mat)
% and set them to GUI.
% [IN]
%               data : GUI data
%    spm_result_file : SPM result file(SPM.mat)

    if exist(spm_result_file, 'file') ~= 2
        return;
    else
        set(data.H.spm_edit, 'String', spm_result_file);

        % Warning off when loading SPM result file
        a = warning('Query', 'MATLAB:unknownElementsNowStruc');
        if strcmpi(a.state, 'on')
            warning('off', 'MATLAB:unknownElementsNowStruc');
            s = load(spm_result_file);
            warning('on', 'MATLAB:unknownElementsNowStruc');
        end
        keys = {s.SPM.xCon.name}';
        Nkeys = length(keys);

        area_keys = cell(Nkeys);
        act_keys  = cell(Nkeys*2);

        postfix = '';
        if isfield(data, 'keys_postfix') && ~isempty(data.keys_postfix)
            postfix = [data.keys_postfix];
        end
        for k=1:Nkeys
            act_keys{k}       = [keys{k}, '_T', postfix];
            act_keys{Nkeys+k} = [keys{k}, '_PS', postfix];
        end
        for k=1:Nkeys
            area_keys{k}      = [keys{k}, '_T', postfix];
        end

        set(data.H.area_key_listbox, 'String', area_keys, 'Value', 1, 'ListboxTop', 1, 'Min', 0, 'Max', length(area_keys));
        set(data.H.act_key_listbox,  'String', act_keys,  'Value', 1, 'ListboxTop', 1, 'Min', 0, 'Max', length(act_keys));
    end

function set_spm_type(H, mode)
% This function changes GUI display according to SPM file type.
    % radio button all off
    radio_group = [H.spm_mni2subj_radiobutton, ...
                   H.spm_mni2mni_radiobutton, ...
                   H.spm_subj2subj_radiobutton];
    set(radio_group, 'Value', 0);
    
    % spm normalization file components
    spm_norm_group = [H.spm_normalization_file_text, ...
                      H.spm_normalization_file_edit, ...
                      H.spm_normalization_file_push];
    
    switch(lower(mode))
        case {'mni2subj'}
            set(H.spm_mni2subj_radiobutton, 'Value', 1);
            set(spm_norm_group, 'Visible', 'on');
        case {'mni2mni'}
            set(H.spm_mni2mni_radiobutton, 'Value', 1);
            set(spm_norm_group, 'Visible', 'off');
        case {'subj2subj'}
            set(H.spm_subj2subj_radiobutton, 'Value', 1);
            set(spm_norm_group, 'Visible', 'off');
        otherwise
            error('Please check function usage.');
    end

function update_exec_push_status(H)
% This function changes Exec button status.(clickable or not)
% [IN]
%    H : GUI component handles
% [OUT]
%    none
    if ~isempty(get(H.spm_edit, 'String')) & ...
       ~isempty(get(H.brain_file_edit, 'String')) & ...
       ~isempty(get(H.area_file_edit, 'String')) & ...
       ~isempty(get(H.act_file_edit, 'String'))
       set(H.exec_push, 'Enable', 'on');
    else
       set(H.exec_push, 'Enable', 'off');
    end
        
function H = exec_fmri(data)
% This function execute vb_job_fmri with set parameter.
% [IN]
%    data : data of fmri gui(@see pa_fmri_create)
% [OUT]
%    none

    H = data.H;

    %%%%%%%%%%%%%%%%%%%%
    % Parameter setting
    %%%%%%%%%%%%%%%%%%%%
    proj_root = get(H.prjroot_edit, 'String');

    % advanced parameters are set here.
    fmri_parm = data.fmri_parm;
    
    % BRAIN file
    fmri_parm.brain_file  = get(H.brain_file_edit, 'String');

    % SPM file
    fmri_parm.SPMmat_file = get(H.spm_edit, 'String');

    % SPM normalization file
    fmri_parm.spm_normalization_file = get(H.spm_normalization_file_edit, 'String');
    
    % SPM mapmode
    if get(H.spm_mni2subj_radiobutton, 'Value')
        fmri_parm.SPMmapmode = 'mni2subj';
    elseif get(H.spm_mni2mni_radiobutton, 'Value')
        fmri_parm.SPMmapmode = 'mni2mni';
    elseif get(H.spm_subj2subj_radiobutton, 'Value')
        fmri_parm.SPMmapmode = 'subj2subj';
    end

    % Postfix of area key and act key
    fmri_parm.keys_postfix = data.keys_postfix;

    % AREA file
    fmri_parm.area_file = get(H.area_file_edit, 'String');

    % ACT file
    fmri_parm.act_file  = get(H.act_file_edit, 'String');

    %%%%%%%%%%%%%%%%%%%%
    % execute job
    %%%%%%%%%%%%%%%%%%%%
    spm_func = which('spm_vol');
    if isempty(spm_func)
        msgbox('Please add SPM8 path to your MATLAB.', 'SPM path is required.');
        return;
    end
    
    str = get(H.exec_push, 'String');
    set(H.exec_push, 'String','Processing...');
    pause(0.01);% for label string update

    vb_job_fmri(proj_root, fmri_parm);

    set(H.exec_push, 'String', 'Exec');
