function varargout = bayes_parm_gui_util(varargin)
% This is a utility function for bayes parameter editor.
%
% Usage: [out1, out2, ...] = bayes_parm_gui_util(command, arg1, arg2,...)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

key = varargin{1};

switch(key)
    case 'filedlg'
        % see function filedlg in this file.
        extensions = varargin{2};
        save_mode  = varargin{3};
        opening_dir = varargin{4};

        [dirname, filename, flg] = filedlg(extensions,...
                                            save_mode,...
                                            opening_dir);
        varargout{1} = dirname;
        varargout{2} = filename;
        varargout{3} = flg;
    case 'filedlg_relative'
        % see function filedlg_relative in this file.
        extensions = varargin{2};
        save_mode  = varargin{3};
        base_dir   = varargin{4};
        if nargin >= 5
            caption    = varargin{5};
        else
            caption = '';
        end

        [dirname, filename, flg] = filedlg_relative(extensions,...
                                            save_mode,...
                                            base_dir, ...
                                            caption);
        varargout{1} = dirname;
        varargout{2} = filename;
        varargout{3} = flg;
    case 'comma_text'
        % see function comma_text in this file.
        text = varargin{2};
        [c_text] = comma_text(text);
        varargout{1} = c_text;
end

function [dirname, filename, flg] = filedlg(extensions,...
                                            save_mode,...
                                            opening_dir, ...
                                            caption);
% This function opens file dialog.
% File dialog returns absolute path.
% [IN] extensions  : selection file extension
%      save_mode   : 0:load mode, 1:save mode
%      opening_dir : default directory
%        caption   : dialog title
% [OUT]dirname     : absolute path
%      filename    : selected filenames(cell array)
%      flg         : dialog end flag(0:cancel, 1:selected)

    persistent last_access_dir;
    persistent history_dir;

    % Default
    if nargin < 3
        % set opening directory
        if length(last_access_dir)
            opening_dir = last_access_dir;
        else
            opening_dir = pwd;
        end
    end
    if nargin < 2
        save_mode = 0;
    end
    if nargin < 1
        extensions = [];
    end
    if ~exist('caption', 'var'), caption = ''; end

    h = file_dialog;
    h.save_mode = save_mode; 
    h.file_extensions = extensions;

    % set directory history.
    hist_num = length(history_dir);
    if ~isempty(history_dir)
        h = set(h, 'hist_dir', history_dir);
    end
    
    % set dialog title
    h = set(h, 'dialog_title', caption);

    % dialog open
    h = set(h, 'current_dir', opening_dir);
    [dirname, filename] = visible(h);

    if isempty(dirname)
        %cancel
        flg = 0;
    else
        %selected
        flg = 1;
        last_access_dir = dirname;
    end

    % save history
    if ~isempty(dirname)
        % check not to duplicate
        idx = strmatch(dirname, history_dir);
        if isempty(idx)
            history_dir{hist_num+1} = dirname;
        end
    end

function [dirname, filename, flg] = filedlg_relative(extensions,...
                                            save_mode,...
                                            base_dir, ...
                                            caption)

% This function opens file dialog.
% File dialog returns relative path.
%
% [IN] extensions  : selection file extension
%      save_mode   : 0:load mode, 1:save mode
%      base_dir    : opening directory
%       caption    : dialog title
% [OUT]dirname     : relative path from base_dir
%      filename    : selected filenames(cell array)
%      flg         : dialog end flag(0:cancel, 1:selected)
    if ~exist('caption', 'var'), caption = ''; end
    [dirname, filename, flg] = filedlg(extensions,...
                                                save_mode,...
                                                base_dir, ...
                                                caption);


    % change path from absolute to relative from base_dir
    if ~isempty(dirname) & ~isempty(base_dir) 
        dirname = vb_relativepath(dirname, base_dir);
        dirname = dirname(1:length(dirname)-1);
    end

function [c_text] = comma_text(text)
% This function makes a comma text.
% text should be (1xn) matrix
%
% [IN]
%    text : 1xn (ex. [5,6,7])
% [OUT]
%    c_text : comma text(ex. '5,6,7')

    c_text = [];
    len = length(text);

    for k=1:len
        if k == len
            c_text = [c_text, sprintf('%.2f', text(k))];
        else
            c_text = [c_text, sprintf('%.2f,', text(k))];
        end
    end
