function [V2ix ,P3 ] = vb_spm_to_cortex(spmx3, spmz, V, PARAM)
% Select the dipoles around the active area in fMRI
%
% ---- Output
%   V2ix   : Vertex indices corresponding to SPM activity points
%   P3     : T-value mapped onto cortical surface (normalized so that
%            maximum=1)
% ---- Input
%   spmx3  : SPM point coordinate [x, y, z] (mm)
%   spmz   : t-value
%   V      : 3D coordinate of vertex number n  (#Vertex*3) 
%
% Originaly written by S.Kajihara
% Ver 1.0  modified by M. Sato  2003-3-15
% 2005-08-30 K. Shibata 
% * scale of t-value is preserved.
% 2008-07-23 Taku Yoshioka (Radius check)
% 2009-05-08 Taku Yoshioka (Radius=0 availale)
% 2011-06-20 taku-y
%  [minor] Progress message was added. 
%
%---- Parameter setting ----
% A vertex point of cortical surface V(n) is chosen as an activity when 
% it is located inside the sphere of radius 'SPM_Radius' with the center
% V(n).
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

SPM_Radius2   = PARAM.SPM_Radius^2 ;% maximum distance for point selection
SPM_zstep	  = PARAM.SPM_zstep  ;	% slice step size in z-direction

% Gauss filter parameter to average SPM t-value
Gauss_radius2 = PARAM.Gauss_radius^2;% half-width of Gaussian filter
Gauss_max	  = PARAM.Gauss_max ;	 % maximum distance for averaging
Gauss_max2	  = PARAM.Gauss_max^2 ;	 % maximum distance for averaging

% Search range of cortex slices in z-direction
zstep = SPM_zstep;  					% slice step size in z-direction
zmin  = min(V(:,3)) - zstep/2; 
zmax  = max(V(:,3)) + zstep/2; 
spmz  = spmz(:);

%---- Parameter setting end----
V2ix  = [];								% Index of active dipoles
P3	  = [];								% Strength of active dipoles

% Loop for slices in z-direction

h       = waitbar(0,'Select dipoles');
prg     = 0;
prg_all = zmax-zmin;
vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));

for zi=zmin:zstep:zmax
  % select dipoles on this slices
  pixz = find( V(:,3) >= zi & V(:,3) < zi+zstep );  

  % select SPM sources near this slice
  spix = find(spmx3(:,3) >= zi - Gauss_max ...
              & spmx3(:,3) <= zi + Gauss_max + zstep);  

  % No SPM point or dipole on this slice
  if min(size(spix))<1 | min(size(pixz))<1, continue; end  
    
  % Find dipoles inside SPM active area
  pixy = find(V(pixz,1) >= min(spmx3(spix,1)) - Gauss_max ...
              & V(pixz,1) <= max(spmx3(spix,1)) + Gauss_max ...
              & V(pixz,2) >= min(spmx3(spix,2)) - Gauss_max ...
              & V(pixz,2) <= max(spmx3(spix,2)) + Gauss_max );  

  % Find neighboring SPM-voxel for each dipole


  % Index for repmat
  indxV = 1:3;
  indxV = indxV( ones( length(spix), 1 ), : );

  Nnext = size(pixy,1);						
  spmxd = spmx3(spix,:);

  % Loop for candidate dipoles in this slice
  for j=1:Nnext,
    
    % Distance between dipole and SPM-voxel
    pix = pixz(pixy(j)); 				% candidate dipole index
    Vp	= V(pix,:);
    d	= spmxd - Vp(indxV);
    dd	= sum(d.^2 , 2 );

    % Radius check (08-07-23 Taku Yoshioka)
    if Gauss_max2<SPM_Radius2, Gauss_max2 = SPM_Radius2; end
        
    % index of dipole near SPM-voxel
    ix	= find( dd <= Gauss_max2 );		% Maximum Averaging radius
    ixx	= find( dd <  SPM_Radius2 );	% Maximum Selection radius
        
    % Check if there are active SPM-voxel in the selection area
    if ~isempty( ixx ),
        
      % Add active dipole index
      V2ix = [V2ix ; pix]; 

      % Gauss-averaged SPM strength for the dipole
      if Gauss_radius2 > 0, 
        Pv = sum(spmz(spix(ix)).*exp(-dd(ix)/Gauss_radius2)) ...
             /sum(exp(-dd(ix)/Gauss_radius2));
      else
        Pv = sum(spmz(spix(ix)))./length(ix);
      end
      P3	 = [P3 ; Pv];
      
    end
  end
  
  waitbar((zi-zmin)/(zmax-zmin));
  for ii=1:15; vb_disp_nonl(sprintf('\b')); end
  vb_disp_nonl(sprintf('%3d %% processed',ceil(100*(prg/prg_all))));
  prg = zi-zmin;
end

% Normalization
% P3	= P3./max(P3); 

close(h);
drawnow;
vb_disp_nonl(sprintf('\n'));

return;