function	G = vb_dipole_basis(V, xx, pick, Qpick, Wsensor, V0, mode)
% Lead field for dipole estimation
%  G = vb_dipole_basis(V, xx, pick, Qpick, Wsensor, V0, mode)
% --- INPUT
%  V(n,:)  : dipole position (3-D coordinate) at n-th vertex
% xx(n,:)  : dipole current direction (unit vector) at n-th vertex
%  pick(k, 1:3) : Sensor coil position  : , 
% Qpick(k, 1:3)	: Sensor coil direction : 
%
% Wsensor(m,n) = n-th coil weight for m-th sensor channel
%   basis(channel,dipole) = Wsensor * basis(coil,dipole)
% --- Optional input
% V0   : Center of spherical model (1 x 3) = [0 0 0] (default)
% mode = 0: Biot-Savart     (default)
%      = 1: Sarvas          
%      = 2: Magnetic dipole 
%      = 3: EEG             
% --- Output
% G : Lead Field (Ndipole, Nsensor)
%
% 2006-12-16 made by M.Sato
% 2007-7-9 modified by M.Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('V0','var') | isempty(V0), V0 = [0 0 0]; end;
if ~exist('mode','var'), mode = 0; end;

% Change center of coordinate
pick = [pick(:,1)-V0(1), pick(:,2)-V0(2), pick(:,3)-V0(3)];
V    = [   V(:,1)-V0(1),    V(:,2)-V0(2),    V(:,3)-V0(3)];

switch mode
case 0
	% Magnetic field by Biot-Savart
	G = vb_dipole_magnetic(V, xx, pick, Qpick ) * Wsensor'; 
case 1
	% Magnetic field by Sarvas
	G = vb_sarvas_basis(V, xx, pick, Qpick) * Wsensor';
case 2
	% Magnetic field by magnetic dipole
	G = vb_dipole_field_magdipole(V, xx, pick, Qpick ) * Wsensor'; 
case 3
	% EEG 3-shell Sphere model
	R = [ 0.87  0.92   1.0 ];
	Rmax  = max(sqrt(sum(V.^2,2)));
	Rpick = mean(sqrt(sum(pick.^2,2)));
	R     = R * Rpick;
	% Conductivity  from innermost to outermost
	sigma = [ 0.62 0.03 0.62 ];
	
	% Electric potential
	G = vb_eeg_legendre(V, xx, pick, R, sigma);
end


