function vb_parm = vb_set_vb_parm(bayes_parm, B, Njact, Njall, v0, sx0)
% VB estimation parameteres 'vb_parm' are set based on 'bayes_parm'
%
% vb_parm = vb_set_vb_parm(bayes_parm, B, [], Njall)
% vb_parm = vb_set_vb_parm(bayes_parm, B, Njact, Njall)
% vb_parm = vb_set_vb_parm(bayes_parm, B, Njact, Njall, [], sx0)
% vb_parm = vb_set_vb_parm(bayes_parm, B, Njact, Njall, v0, sx0)
%
% --- input
% B : MEG signal
% Njact : # of focal current
% Njall : # of global current
% v0  : baseline activity
% sx0 : noise variance
%
% [history]
% 2005/08/22 O. Yamashita ver.30b 
% 2006/08/04 M. Sato
% 2011-11-16 taku-y
%  [major] Offset of time index was corrected. 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

switch nargin
    case 1
        help vb_set_vb_parm;
    case 2
        Njact = [];
        Njall = [];
        v0  = 1;
        sx0 = 1;
    case 3 
        Njall = [];
        v0  = 1;
        sx0 = 1;
    case 4
        v0  = 1;
        sx0 = 1;
    case 5
        sx0 = 1;
end

if isempty(v0)
    v0  = 1;
end
if isempty(sx0)
    sx0 = 1;
end

if ~isfield(bayes_parm, 'Ta0')
    bayes_parm.Ta0 = 0; % No fMRI bias
end

vb_parm.Ntrain   = bayes_parm.Ntrain;
vb_parm.Nskip	 = bayes_parm.skip;
vb_parm.a_min	 = bayes_parm.a_min;
vb_parm.a_max	 = bayes_parm.a_max;
vb_parm.Fdmin	 = bayes_parm.Fdmin;

vb_parm.update_v  = bayes_parm.update_v;
vb_parm.update_sx = bayes_parm.update_sx;

if isfield(bayes_parm, 'Npre_train')
	% Number of iteration using original VB-update rule
	% for stable estimation
	vb_parm.Npre_train = bayes_parm.Npre_train;
else
	vb_parm.Npre_train = 0;
end

if isfield(bayes_parm,'cont_pr'), 
	vb_parm.cont_pr = bayes_parm.cont_pr;
else 
	vb_parm.cont_pr = 0; 
end


% fields must be set before estimation
vb_parm.Nvact = [];        % number of vertex points in focal window
vb_parm.Njact = [];        % number of current parameters in focal window
vb_parm.Norient = [];      % number of dipole orientation (1,2,3)
vb_parm.Norient_var = [];  % number of orientation to be estimated 
vb_parm.Nsession = [];     % number of session
vb_parm.Njall = [];        % number of currents parameters in global window
vb_parm.Nvall = [];        % number of vertex points in global window
vb_parm.Norient_all = [];  % number of dipole orientation in global window     
vb_parm.Ntrials = [];      % number of trials in each session 
vb_parm.Nsensors = [];     % number of meg sensors in each session
vb_parm.Tsample = [];      % number of total time points
vb_parm.Twindow = [];      % start and end time index of time windows 
vb_parm.Nwindow = [];      % number of time windows
vb_parm.sx0 = [];          % initial value of observation noise variance
vb_parm.v0  = [];          % initial value of global window variance
vb_parm.Tv0 = [];          % confidence parameter of global window variance
vb_parm.a0 = [];           % initial value of focal window variance
vb_parm.Ta0 = [];          % confidence parameter of focal window variance

vb_parm = set_session(vb_parm, B);
vb_parm = set_twin(bayes_parm, vb_parm);
vb_parm = set_global(bayes_parm, vb_parm, Njall);
vb_parm = set_focal(bayes_parm, vb_parm, Njact);

vb_parm = set_prior_def(bayes_parm,vb_parm,v0,sx0);

check_bayes_parm(bayes_parm,vb_parm);

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%
%% inner functions
%%

%%%%% session, senosors, trials
function [vb_parm] = set_session(vb_parm,B)
Nsession = length(B);
for n = 1: length(B)
   [Nsensors(n),tmp,Ntrials(n)]  = size(B{n});
end
vb_parm.Nsession = Nsession;
vb_parm.Ntrials = Ntrials;
vb_parm.Nsensors = Nsensors;

return

%%%%% time window
function [vb_parm] = set_twin(bayes_parm,vb_parm)
Tsample = bayes_parm.twin_meg(2) - bayes_parm.twin_meg(1) + 1;

% offset
if length(bayes_parm.Tnext) > 1, 
	Tnext = bayes_parm.Tnext-bayes_parm.twin_meg(1)+1;
elseif length(bayes_parm.Tnext) == 1
    Tnext = bayes_parm.Tnext;
end

Twindow = vb_calc_timewindow(Tsample,bayes_parm.Tperiod,Tnext);
vb_parm.Tsample = Tsample;
vb_parm.Twindow = Twindow;
vb_parm.Nwindow = size(Twindow,1);

return

%%%%% global window parameters
function [vb_parm] = set_global(bayes_parm,vb_parm,Njall)
[basis,L]=vb_load_basis(bayes_parm.basisfile_global{1});

vb_parm.Norient_all = L;

if ~isempty(Njall)
	vb_parm.Njall = Njall;
	vb_parm.Nvall = Njall/L;
else
	vb_parm.Njall = 0;
	vb_parm.Nvall = 0;
end

return

%%%%% focal window parameters (area file)
function [vb_parm] = set_focal(bayes_parm,vb_parm,Njact)
[tmp,L]=vb_load_basis(bayes_parm.basisfile{1});

vb_parm.Norient = L;

if ~isempty(Njact)
    vb_parm.Njact = Njact;
	vb_parm.Nvact = Njact/L;
else
	vb_parm.Njact = 0;
    vb_parm.Nvact = 0;
end

return

%%%%% Default prior parameter
function	vb_parm = set_prior_def(bayes_parm,vb_parm,v0,sx0)

if ~isfield(bayes_parm,'soft_mode'),
	vb_parm.soft_mode = 0;
else
	vb_parm.soft_mode = bayes_parm.soft_mode;
end

if bayes_parm.variance_orientation == ON 
	vb_parm.Norient_var = vb_parm.Norient;
	
	% ʬ = sin( tan_var * ʿˡγʿ) <= var_max
	% tangential variance coefficient
	if isfield(bayes_parm,'tan_var')
		vb_parm.tan_var = bayes_parm.tan_var;
	else
		vb_parm.tan_var = 1;
	end
	
	% tangential variance max
	if isfield(bayes_parm,'var_max')
		vb_parm.var_max = bayes_parm.var_max;
	else
		vb_parm.var_max = 1/sqrt(2);
	end
else
	vb_parm.Norient_var = 1;
end

vb_parm.sx0 = sx0;

vb_parm.v0  = v0 ;
vb_parm.a0  = v0;

vb_parm.Tv0 = bayes_parm.Tv0;
vb_parm.Ta0 = bayes_parm.Ta0;

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% Check bayes_parm in the case of soft normal constraint %%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function	result = check_bayes_parm(bayes_parm,vb_parm)

result = FAILURE;

% orientation estimate with soft constraint
if isfield(bayes_parm,'soft_mode') & bayes_parm.soft_mode == 1,
	
	if bayes_parm.variance_orientation == OFF; 
		fprintf('In the case of soft constraint mode\n')
		error('variance_orientation should be ON');
	end
	
	if vb_parm.Norient ~= 3 | vb_parm.Norient_var ~= 3;
		fprintf('In the case of soft constraint mode\n')
		error('Norient should be 3 ');
	end
	
%	if ~isempty(bayes_parm.reduce) & bayes_parm.reduce ~= 1,
%		error('bayes_parm.reduce should be empty ');
%	end
	
end


result = SUCCESS;

return
