function [KW,KW_ext] = vb_invfilter_z(a_inv,G,Cov,jx_act,e_inv,Gext)
% Inverse filter for Z-current reconstruction
%
% --- Syntax
% [KW,KW_ext] = vb_invfilter_z(a_inv,G,Cov,jx_act,e_inv,Gext)
%
% --- Input
% a_inv : A    :  variance of active current                    
% G     : Ga : leadfield of active current (smoothed) G0*Wa  
% Cov   : Sensor noise covariance matrix
%
% --- Optional input
% jx_act : selected vertex index for current estimation
%          relative index within a_inv
% e_inv  : 
%
% ---Output
% KW    : Inverse filter for active current
% KW_ext: Inverse filter for 
%
% --- Current reconstruction
% Zact = (KW  * Bt);
% Original vertex current can be obtained by applying spatial smoothing filter
% Jact = Wact * Zact
% Note: smoothing filter should not be applied to extra dipole currents. 
%
% --- History
% 2008-7-1 M.Sato
% 2008-08-19 Taku Yoshioka (support extra dipole)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Nact   # of estimated variance in focal area
% Njact  # of current in focal area
% # of active current = Njact = Ratio * (# of a_inv)

if nargin<5, e_inv = []; end

Njact = size(G,2);
Nact  = length(a_inv);
Ratio = Njact/Nact;

if Ratio > 1
  a_inv  = repmat(a_inv , [Ratio 1]);
end

if isempty(e_inv), 
  AGt = vb_repmultiply(G', a_inv);   % = A*Ga'
  EGt = [];                       % E*Gext'
  SB_inv = pinv( G * AGt + Cov ); % SB^(-1) = [Ga*A*Ga' + Cov]^{-1}
else
  AGt = vb_repmultiply(G', a_inv);    % = A*Ga'
  EGt = vb_repmultiply(Gext', e_inv); % E*Gext'
  SB_inv = pinv( G * AGt + Gext * EGt + Cov ); 
end

if exist('jx_act','var'), 
  if Ratio > 1
    jx_act = repmat(jx_act(:),1,Ratio) ...
             + repmat([0:Ratio-1]*Nact,length(jx_act),1);
  end
  jx_act = jx_act(:);
  AGt = AGt(jx_act,:);
end

%%%% Inverse filter
KW  = AGt * SB_inv;            % A*Ga'*(SB)^{-1}
if ~isempty(EGt), KW_ext = EGt * SB_inv; 
else KW_ext = []; end
