function [Z, ix_roi] = vb_calc_roi_current(current_file, area_file, brain_file, mode)
% Calculate ROI-current according to area file
%
% - Input
%   current_file : Current file (.curr.mat)
%   area_file : Area file (.area.mat) defining ROIs
%   brain_file : Brain file (.brain.mat)
%   mode : 1 -> Average (default)
%          2 -> First principal component
%          3 -> Maximum current
%
% - Output
%   Z : ROI-current (roi x time x trial)
%   ix_roi : Index of center vertex in each ROI
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Load data
[~, current] = vb_load_current(current_file);
load(area_file, 'Area')
[V, F, xx, inf_C] = vb_load_cortex(brain_file);

if ~exist('mode', 'var')
    mode = 1;
end

% Calculate ROI-current
Nroi = length(Area);
[~, Nt, Ntr] = size(current);
Z = zeros(Nroi, Nt, Ntr);
switch mode
    case 1% Average
        for roi = 1:Nroi
            ix = Area{roi}.Iextract;
            Z(roi, :, :) = mean(current(ix, :, :), 1);
        end
    case 2% Principal component
        for roi = 1:Nroi
            ix = Area{roi}.Iextract;
            Nix = length(ix);
            c = reshape(current(ix, :, :), Nix, Nt*Ntr)';
            [u, s] = svd(c, 0);
            us = u(:, 1)*s(1, 1);
            mc = mean(current(ix, :, :), 1);
            mc = mc(:)';
            us = us*sign(mc*us);
            Z(roi, :, :) = reshape(us', 1, Nt, Ntr);
        end
    case 3% Maximum current
        for roi = 1:Nroi
            ix = Area{roi}.Iextract;
            Nix = length(ix);
            c = reshape(current(ix, :, :), Nix, Nt*Ntr);
            pc = sum(c.^2, 2);
            [~, ix2] = max(pc);
            Z(roi, :, :) = current(ix(ix2), :, :);
        end
end

% Obtain index of center vertex in each ROI
ix_roi = zeros(Nroi, 1);
for roi = 1:Nroi
    ix = Area{roi}.Iextract;
    Nix = length(ix);
    center = mean(V(ix,:), 1);
    distance = sum((V(ix, :)-repmat(center, Nix, 1)).^2, 2);
    [~, b] = min(distance);
    ix_roi(roi) = ix(b);
end

% Check ix_roi
if 0
    figure
    color = hsv(Nroi);
    [~, ix] = sortrows(rand(Nroi, 1));
    color = color(ix, :);
    plot_parm = vb_set_plot_parm;
    vb_plot_cortex(plot_parm, V, F, inf_C);
    for roi = 1:Nroi
        ix = Area{roi}.Iextract;
        Nix = length(ix);
        for i = 1: Nix
            plot3(V(ix(i), 1), V(ix(i), 2), V(ix(i), 3), 'o', 'markersize', 5,...
                'markerfacecolor', color(roi, :), 'markeredgecolor', color(roi, :))
        end
        plot3(V(ix_roi(roi), 1), V(ix_roi(roi), 2), V(ix_roi(roi), 3), 'o', 'markersize', 7,...
            'markerfacecolor', color(roi,:), 'markeredgecolor', [1 1 1])
    end
    axis off
end



