function [Z,W,Jinfo] = vb_calc_current(varargin)
% Current reconstruction using Bayesian inverse filter
% (VBMEG public function)
%
% [syntax]
% [Z,W,Jinfo] = vb_calc_current(VBfilt,bayes_parm,curr_parm)
% [Z,W,Jinfo] = vb_calc_current(VBfilt,bexp,curr_parm)
% [Z,W,Jinfo] = vb_calc_current(proj_root,VBfilt,bayes_parm,curr_parm)
% [Z,W,Jinfo] = vb_calc_current(proj_root,VBfilt,bexp,curr_parm)
%
% [input]
% proj_root : <optional> <<string>> VBMEG project root directory. 
% VBfilt    : <<struct>> Inverse filter and related variables.
% bayes_parm: <<struct>> Parameters for variational Bayes method. Fields
%             of this parameter is the same with that of job_vb.m. 
% bexp      : <<cell>> Cell of MEG data. If 'bexp' is given instead of
%             struct 'bayes_parm', the following variables, required for
%             current reconstruction will be set to ones in struct
%             'VBfilt' (see example): 
%                 Ntrial
%                 Tsample
%                 Twindow
% curr_parm : <<struct>> Current reconstruction parameters. 
% --- fields of curr_parm
%  overlap_mode : <<int>> Specify how cortical current is estimated. 
%   = 0 : current is averaged over overlapped time window. 
%   = 1 : current is not averaged for overlapped window. 
%         current time series of each time windows 
%         are concatenated sequentially for spectral analysis
%  ix_area      : <optional> <<vector>> Vertex indices to calculate
%                 cortical current. If 'ix_area' is empty or not given,
%                 currents in the active region are calculated. 
%  trial_average: <<bool>> If true, average current over all sessions and
%                 trials is calculated. 
%  ix_trial     : <<vector>> Trial indices for which currents are
%                 reconstructed.
%  tsubsamp     : <optional> <<bool>> Specify subsampled time index
%                 If 'tsubsamp' is empty or not given, time subsampling
%                 is not done. 
% ---
%
% [output]
% Z    : <<matrix>> Z-current ([I,Nsample,Ntrial]). 
% W    : <<matrix>> Spatial smoothing matrix. 
% Jinfo: <<struct>> Information of estimated current. 
% --- fields of Jinfo
%  ix_act   : <<vector>> Cortical vertex indices of Z-current. 
%  ix_act_ex: <<vector>> Cortical vertex indices of J-current.
% ---
%
% [example]
% >> bexp{1} = vb_load_meg_data(meg_file); % assume single session
% >> load(filter_file,'VBfilt','bayes_parm'); % .vbfilt.mat
% >> vb_struct2vars(bayes_parm,{'twin_meg','Tperiod','Tnext'});
% >> bexp{1} = bexp{1}(:,twin_meg(1):twin_meg(2),:);
% >> VBfilt.Ntrial(1) = size(bexp{1},3);
% >> VBfilt.Tsample = twin_meg(2)-twin_meg(1)+1;
% >> VBfilt.Twindow = vb_calc_timewindow(VBfilt.Tsample,Tperiod,Tnext);
% >> curr_parm.overlap_mode  = 0;
% >> curr_parm.ix_area       = [];
% >> curr_parm.trial_average = true;
% >> curr_parm.ix_trial      = [];
% >> curr_parm.tsubsamp      = [];
% >> [Z,W,Jinfo] = vb_calc_current(proj_root,VBfilt,bexp,curr_parm);
%
% [history]
% 2010-06-28 Taku Yoshioka
% 2010-07-12 Taku Yoshioka
%  'trial_average' support.
% 2010-09-29 Taku Yoshioka
%  New usage of function implemented ('bexp' given instead of
%  'bayes_parm'). 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% Verbose level
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
global vbmeg_inst;
const = vb_define_verbose;
if isfield(vbmeg_inst,'verbose_level'), 
  verbose_level = vbmeg_inst.verbose_level;
else
  verbose_level = const.VERBOSE_LEVEL_NOTICE;
end;

%
% Check input arguments
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if length(varargin) == 3
  proj_root = [];
  VBfilt = varargin{1};
  curr_parm = varargin{3};
  if isstruct(varargin{2}), 
    bayes_parm = varargin{2};
  else
    bexp = varargin{2}; % 'bexp' is a cell array.
  end;
elseif length(varargin) == 4
  proj_root = varargin{1};
  VBfilt = varargin{2};
  if isstruct(varargin{3}), 
    bayes_parm = varargin{3};
  else
    bexp = varargin{3}; % 'bexp' is a cell array.
  end;
  curr_parm = varargin{4};
else
  error('Error: invalid usage of vb_calc_current.');
end;

vb_struct2vars(curr_parm,{'overlap_mode','ix_area','trial_average', ...
                    'ix_trial','tsubsamp'});

if exist('bayes_parm'), 
  if ~isempty(proj_root);
    bayes_parm_abs = vb_parm_absolute_path(proj_root,bayes_parm);
  else
    bayes_parm_abs = bayes_parm;
  end;
end;

%
% MEG data preparation
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if exist('bayes_parm_abs'), 
  [bexp,Ntrial,Nch,Tsample,Twindow] ...
      = vb_megdata_preparation(bayes_parm_abs);
  Nsession = length(bexp);	% Number of sessions
else
  Tsample = VBfilt.Tsample;
  Twindow = VBfilt.Twindow;
  Ntrial = VBfilt.Ntrial;
  Nsession = length(bexp);	% Number of sessions
end;

%
% Area index in which current is calculated
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
ix_act = VBfilt.ix_act;
ix_act_ex = VBfilt.ix_act_ex;

if ~isempty(ix_area),
  % Select vertex index 'ix_area' within the active current region
  [jx_area_ex, ix_area_ex] = vb_index2indexsub(ix_area, ix_act_ex);
else
  jx_area_ex = 1:length(ix_act_ex);
end

W = VBfilt.W(jx_area_ex,:);
jx_act = find(sum(W,1)>0);
W = W(:,jx_act);

ix_act = ix_act(jx_act); % active index of Z-current
ix_act_ex = ix_act_ex(jx_area_ex); % active index of J-current

% # of active vertex
Njact_area = length(jx_act);

% # of extra dipoles
if isempty(VBfilt.KW_ext), Njext = 0;
else Njext = size(VBfilt.KW_ext{1},1); end;

Jinfo.ix_act = ix_act;
Jinfo.ix_act_ex = ix_act_ex;

%
% Temporal smoothing window weight
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(tsubsamp),
  tsubsamp = 1:Tsample;
end;

[Tweight ,Tindex, Nindex, Nsample] = ...
    vb_overlapped_timewindow_weight(Twindow, Tsample, tsubsamp, overlap_mode);

Nwindow = length(Nindex);   	% # of time window
vb_disp(['Number of time windows : ' num2str(Nwindow)]);

if overlap_mode == 1,
  vb_disp('Non-overlapped concatenation mode'); 
end;

%
% Calculate Z-current
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Ntrial_all = 0;

% Session loop
for n=1:Nsession
  % Trial average
  if trial_average, 
    bexp{n} = mean(bexp{n},3); 
    ix_trial(n) = 1;
  elseif isempty(curr_parm.ix_trial), 
    ix_trial(n) = VBfilt.Ntrial(n);
  end;
  
  % Number of trials
  Ntrial(n) = length(ix_trial(n));

  % Get inverse filter for all time window
  for j=1:Nwindow
    Nid = Nindex{j};	% index in the total subsampled data
    %KW{j} = VBfilt.KW(:,:,j,n);
    KW{j} = VBfilt.KW(jx_act,:,j,n);
  end;

  % Trial loop
  for m=ix_trial(n)
    Zact = zeros(Njact_area,Nsample);
		
    for j=1:Nwindow
      % Subsampling time index
      Tid = Tindex{j};	% subsampled time index
      Nid = Nindex{j};	% index in the total subsampled data

      if isempty(Nid), continue; end;
			
      % Time window smoothing
      Bt = vb_repmultiply(bexp{n}(:,Tid,m), Tweight{j});
      Zact(:,Nid) = Zact(:,Nid) + (KW{j} * Bt);
    end; % Timewindow loop END
    
    Ntrial_all = Ntrial_all+1;
    Z(:,:,Ntrial_all) = Zact;
  end; % Trial loop END
  
  vb_disp_nonl('.');
end; % Session loop END
vb_disp_nonl(sprintf('\n'));

Jinfo.Ntrial   = Ntrial;
Jinfo.Nsession = Nsession;

%
% Calculate extra-dipole current (not implemented: 2010-06-28)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

return;
