function [data, ch_info] = vb_msrmnt_make_trial_data(meeg_file, proc_spec, new_meeg_file)
% Make trial data files by chopping continuous MEG or EEG data
% [usage]
%   [data, ch_info] = vb_msrmnt_make_trial_data(meeg_file, proc_spec, new_meeg_file)
% [input]
%    meeg_file : <required> <<file>>
%                Continuous format MEG data file(.meg.mat) or
%                Continuous format EEG data file(.eeg.mat)
%    proc_spec : <required> <<struct>>
%              :  struct defined process specifications which has fields 
%              :  as follows:
%              :
%      .trig_file : <conditionally required> <<file>>
%                 :  trigger file(.trig.mat)
%                 :  this file is expected to have data as follows:
%                 :   .trig [1 x Ntrial]
%                 :   .parm.data_file (without path) : base file
%                 :   .parm.Pretrigger_ms  : pretrigger [msec]
%                 :   .parm.Posttrigger_ms : posttrigger [msec]
%                 :   .parm.fsamp : sampling frequency [Hz]
%                 :  if this is invalid and .proc_parm field is not
%                 :  specified, it is error.
%      .internal  : <optional>
%                   = true  : single file format(MEG/EEG). [default]
%                   = false : multiple file format(MAT+binary dir)(only MEG).
%      .proc_parm : <optional> <<struct>>
%                 :  .Pretrigger_ms  : Pretrigger period   [msec]
%                 :  .Posttrigger_ms : Posttrigger period  [msec]
%                 :  .sample_freq    : sampling frequency [Hz]
%                 :  .trigger        : list of trigger samples
%                 :                  : [1 x Ntrial]
%                 :    if these are specified, 
%                 :    they overwrite the setting values of trig_file
%   new_meeg_file : <required> <<file>>
%                   Trial format MEG data file(.meg.mat) or
%                   Trial format EEG data file(.eeg.mat)
% [output]
%           : these are the same as the return values of vb_load_meg_data
%      data : chopped data [Nchannel x Nsample x Ntrial]
%   ch_info : channel information struct
%           :  .Active
%           :  .Name
%           :  .ID
%           :  .Type
% [note]
%
% [history]
%   2009-07-24 (Sako) initial version
%   2009-09-01 (Sako) substituted vb_load_measurement_info for vb_load_meg_info
%   2016-11-28 (rhayashi) Change specification(meeg_file, new_meeg_file)
%   2018-08-22 (Takeda) Modified so that cond_id is inherited 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('proc_spec', 'var'), proc_spec = []; end
if ~exist('meeg_file', 'var'), meeg_file = ''; end
if ~exist('new_meeg_file', 'var'), new_meeg_file = ''; end

[meeg_file, proc_spec, new_meeg_file] = ...
    inner_check_arguments(meeg_file, proc_spec, new_meeg_file);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% default values
[base_file_def, trigger_def, cond_id_def, pretrigger_def, posttrigger_def, ...
  sample_freq_def] = inner_read_trigger_file(proc_spec.trig_file);

% substitute values
[base_file_sub, trigger_sub, cond_id_sub, pretrigger_sub, posttrigger_sub, ...
  sample_freq_sub] = inner_read_proc_parm(meeg_file, proc_spec.proc_parm);

if ~isempty(trigger_sub)
  trigger = trigger_sub;
else
  trigger = trigger_def;
end

if ~isempty(cond_id_sub)
  cond_id = cond_id_sub;
else
  cond_id = cond_id_def;
end

if ~isempty(pretrigger_sub)
  pretrigger_ms = pretrigger_sub;
else
  pretrigger_ms = pretrigger_def;
end

if ~isempty(posttrigger_sub)
  posttrigger_ms = posttrigger_sub;
else
  posttrigger_ms = posttrigger_def;
end

% sample frequency for triger file
if ~isempty(sample_freq_sub)
  sample_freq = sample_freq_sub;
else
  sample_freq = sample_freq_def;
end


info = vb_load_measurement_info(meeg_file);
Measurement = vb_info_get_measurement(info);

% --- MEG
if strcmp(Measurement, 'MEG')
    [f_, p_, e_] = vb_get_file_parts(new_meeg_file);
    new_spec.trigger = trigger;
    new_spec.cond_id = cond_id;
    new_spec.pretrigger_ms = pretrigger_ms;
    new_spec.posttrigger_ms = posttrigger_ms;
    new_spec.sample_freq = sample_freq;
    new_spec.root_dir = f_;
    new_spec.new_file = [p_, e_];
    new_spec.new_bin_dir = proc_spec.new_bin_dir;
    [data, ch_info] = vb_megfile_make_trial_data(meeg_file, new_spec);
else
    % EEG case
    
    % --- make load_spec

    % sample frequency for base_file
    info = vb_load_measurement_info(meeg_file);
    fs   = vb_info_get_sampling_frequency(info);

    % Convert trigger time index for fs [Hz]
    load_spec.Trigger = round(trigger * (fs/sample_freq));

    % convert trigger data from msec to sample
    load_spec.Pretrigger  = ceil(pretrigger_ms  *(fs/1000));
    load_spec.Posttrigger = ceil(posttrigger_ms *(fs/1000));

    load_spec.ChannelType = 'ALL';
    load_spec.ActiveChannel = false;

    [data, ch_info] = vb_load_meg_data(meeg_file, load_spec, new_meeg_file);
    
    % Add cond_id to EEGinfo
    load(new_meeg_file, 'EEGinfo');
    EEGinfo.cond_id = cond_id;
    vb_save(new_meeg_file, 'EEGinfo');
end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meeg_file, proc_spec, new_meeg_file] = ...
    inner_check_arguments(meeg_file, proc_spec, new_meeg_file)
func_ = mfilename;

if isempty(meeg_file)
  error('(%s) meeg_file is a required parameter', func_);
end
if exist(meeg_file, 'file') ~= 2
  error('(%s) Specified meeg_file not found.', func_);
end    
if isempty(strfind(meeg_file, '.meg.mat')) && ...
   isempty(strfind(meeg_file, '.eeg.mat'))
  error('(%s) meeg_file is not MEG(.meg.mat) or EEG(.eeg.mat) file.', func_);
end

if isempty(proc_spec)
  error('(%s) proc_spec is a required parameter', func_);
end

if isempty(new_meeg_file)
  error('(%s) new_meeg_file is a required parameter', func_);
end

% --- proc_spec
%  .trig_file
%  .internal
%  .proc_parm

if ~isfield(proc_spec, 'trig_file')
  proc_spec.trig_file = '';
end

if ~isfield(proc_spec, 'internal')
  proc_spec.internal = true;
end

if proc_spec.internal == false
  [p_, f_, e_] = vb_get_file_parts(new_meeg_file);
  new_bin_dir = f_(1:end-4); % f_ : -4 means '.eeg' or '.meg'
  proc_spec.new_bin_dir = ['./' new_bin_dir];
else
  proc_spec.new_bin_dir = '';
end

if ~isfield(proc_spec, 'proc_parm')
  proc_spec.proc_parm = [];
end

if isempty(proc_spec.trig_file) && isempty(proc_spec.proc_parm)
  error('(%s) either trig_file or proc_parm is required', func_);
end
return;
%
% --- end of inner_check_arguments()


% --- inner_read_trigger_file()
%
function [base_file, trigger, cond_id, pretrigger, posttrigger, sample_freq] = ...
  inner_read_trigger_file(trig_file)

func_ = mfilename;

base_file = '';
trigger = [];
cond_id = [];
pretrigger = [];
posttrigger = [];
sample_freq = 1024;

if isempty(trig_file)
  return;
end

if exist(trig_file, 'file') ~= 2
  error('(%s) cannot find trig_file : %s', func_, trig_file);
end

trig = load(trig_file);
if isfield(trig, 'trig') && ~isempty(trig.trig)
  trigger = trig.trig;
end
if isfield(trig, 'cond_id') && ~isempty(trig.cond_id)
  cond_id = trig.cond_id;
end

if isfield(trig, 'parm') && ~isempty(trig.parm)
  if isfield(trig.parm, 'data_file') && ~isempty(trig.parm.data_file)
    base_file = trig.parm.data_file;
  end

  if isfield(trig.parm, 'Pretrigger_ms')
    pretrigger = trig.parm.Pretrigger_ms;
  end

  if isfield(trig.parm, 'Posttrigger_ms')
    posttrigger = trig.parm.Posttrigger_ms;
  end
  
  if isfield(trig.parm, 'fsamp')
    sample_freq = trig.parm.fsamp;
  end
end
return;
%
% --- end of inner_read_trigger_file()


% --- inner_read_proc_parm()
%
function [base_file, trigger, cond_id, pretrigger, posttrigger, sample_freq] = ...
  inner_read_proc_parm(meeg_file, parm)

base_file = '';
trigger = [];
cond_id = [];
pretrigger = [];
posttrigger = [];
sample_freq = [];

if isempty(parm)
  return;
end

if isfield(parm, 'trigger') && ~isempty(parm.trigger)
  trigger = parm.trigger;
end
if isfield(parm, 'cond_id') && ~isempty(parm.cond_id)
  cond_id = parm.cond_id;
end

[p_, f_, e_] = vb_get_file_parts(meeg_file);
base_file = [f_, e_];

if isfield(parm, 'Pretrigger_ms')
  pretrigger = parm.Pretrigger_ms;
end

if isfield(parm, 'Posttrigger_ms')
  posttrigger = parm.Posttrigger_ms;
end

if isfield(parm, 'sample_freq')
  sample_freq = parm.sample_freq;
end
return;
%
% --- end of inner_read_proc_parm()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
