function [loadspec, result] = vb_loadspec_check(loadspec, msrmntfile)
% check LoadSpec data
%
% [usage]
%   [loadspec, result] = vb_loadspec_check(loadspec, msrmntfile)
%
% [input]
%   loadspec : <optional> <<struct>> LoadSpec
%            :  .ChannelName : {Nch x 1 cell} channel name list 
%            :    [all channels]
%            :  .ChannelSwitch : <<boolean>> ChannelName role
%            :    [true] ) to read
%            :    false  ) to omit
%            :  .ChannelType : <optional> <<string>> channel type ['MEG(EEG)']
%            :               :  'MEG'       : MEG channel data
%            :               :  'EEG'       : MEG channel data
%            :               :  'EXTRA'     : extra channel data
%            :               :  'REFERENCE' : reference channel data (only MEG)
%            :               :  'AXIAL'     : AxialGradioMeter (only MEG)
%            :               :  'PLANAR'    : PlanarGradioMeter (only MEG)
%            :               :  'MAG_X'     : Magnetometer OPM channel data toward X-axis (only QZFM device)
%            :               :  'MAG_Y'     : Magnetometer OPM channel data toward Y-axis (only QZFM device)
%            :               :  'MAG_Z'     : Magnetometer OPM channel data toward Z-axis (only QZFM device)
%            :               :  'ALL'       : all the channels
%            :               :  - case insensitive
%            :  .Trigger : [Ntrigger x1 double] trigger list [1]
%            :  .Pretrigger : [x1 double] number of sample before trigger [0]
%            :  .Posttrigger : [x1 double] number of sample after trigger
%            :    [(lenght of sample)-1]
%            :  .TrialNumber : [x1 double] trial index list [all trial]
%            :  .ActiveChannel : <<boolean>> active channel filter switch
%            :                 :  if active channel is valid [true]
%            :                 :  if active channel is invalid [false]
%            :  .ActiveTrial   : <<boolean>> active trial filter switch
%            :                 :  if active trial is valid [true]
%            :                 :  if active trial is invalid [false]
% msrmntfile : <required> <<file>> MEG-MAT or EEG-MAT file
%
% [output]
%   loadspec : LoadSpec struct after check
%     result : <<integer>> error code
%            :  0) success
%            :  1) error - bad msrmntfile
%
% [note]
%   this function does not abort but return error code as 'result'
%
% [history]
%   2008-02-29 (Sako) initial version
%   2008-04-16 (Sako) added ActiveChannel field 
%   2008-04-28 (Sako) added loadtype
%   2008-06-04 (Sako) changed default values of .ActiveChannel and .ActiveTrial
%   2008-06-05 (Sako) added ChannelType field
%   2009-07-14 (Sako) modified default value of .TrialNumber
%   2009-07-24 (Sako) changed the default value of .ChannelType to 'ALL'
%   2009-07-27 (Sako) modified ChannelType by using new vb_define_const
%   2011-05-31 (Sako) modified to convert device character to be upper case
%   2022-06-03 (k_suzuki) added OPM format
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('loadspec', 'var'), loadspec = []; end
if ~exist('msrmntfile', 'var'), msrmntfile = ''; end
[loadspec, msrmntfile] = inner_check_arguments(loadspec, msrmntfile);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
[measurement, device] = vb_load_device(msrmntfile);
Measurement = upper(measurement);

const = vb_define_const(Measurement);

if isempty(loadspec.ChannelType)
    loadspec.ChannelType = const.DATA_TYPE_MAIN;
end

channel_name = [];

% --- obtain basic information
switch Measurement
    case 'MEG'
        switch device
            % SQUID-devices
            case {'YOKOGAWA', 'NEUROMAG'}
                switch loadspec.ChannelType
                    case const.DATA_TYPE_MAIN
                        channel_name = vb_megfile_get_channel_label_meg(msrmntfile);
                    case const.DATA_TYPE_EXTRA
                        channel_name = vb_megfile_get_channel_label_extra(msrmntfile);
                    case const.DATA_TYPE_REFERENCE
                        channel_name = vb_megfile_get_channel_label_refmg(msrmntfile);
                    case const.DATA_TYPE_AXIAL
                        channel_name = vb_megfile_get_channel_label_axial(msrmntfile);
                    case const.DATA_TYPE_PLANAR
                        channel_name = vb_megfile_get_channel_label_planar(msrmntfile);
                    case const.DATA_TYPE_ALL
                        channel_name = vb_megfile_get_channel_label_whole(msrmntfile);
                end
                
            % OPM-device
            case 'QZFM'
                switch loadspec.ChannelType
                    case const.DATA_TYPE_MAIN
                        channel_name = vb_megfile_get_channel_label_meg(msrmntfile);
                    case const.DATA_TYPE_EXTRA
                        channel_name = vb_megfile_get_channel_label_extra(msrmntfile);
                    case const.DATA_TYPE_REFERENCE
                        channel_name = vb_megfile_get_channel_label_refmg(msrmntfile);
                    case const.DATA_TYPE_MAG_X
                        channel_name = vb_megfile_get_channel_label_magx(msrmntfile);
                    case const.DATA_TYPE_MAG_Y
                        channel_name = vb_megfile_get_channel_label_magy(msrmntfile);
                    case const.DATA_TYPE_MAG_Z
                        channel_name = vb_megfile_get_channel_label_magz(msrmntfile);
                    case const.DATA_TYPE_ALL
                        channel_name = vb_megfile_get_channel_label_whole(msrmntfile);
                end
        end
        
        % Common procedure for MEG
        meginfo = vb_megfile_load_meginfo(msrmntfile);
        trial_idx = vb_info_get_trial_data(meginfo, [], 'number');
        n_trial = vb_info_get_Nrepeat(meginfo);
        
    case 'EEG'
        eeginfo = vb_eegfile_load_eeginfo(msrmntfile);
        
        switch loadspec.ChannelType
            case const.DATA_TYPE_MAIN
                channel_name = vb_eeginfo_get_channel_label(eeginfo);
            case const.DATA_TYPE_EXTRA
                channel_name = vb_eeginfo_get_channel_label_extra(eeginfo);
            case const.DATA_TYPE_ALL
                ch_name_eeg = vb_eeginfo_get_channel_label(eeginfo);
                ch_name_ext = vb_eeginfo_get_channel_label_extra(eeginfo);
                channel_name = [ch_name_eeg;ch_name_ext];
        end
        
        trial_idx = vb_info_get_trial_data(eeginfo, [], 'number');
        n_trial = vb_info_get_Nrepeat(eeginfo);
        
    otherwise
        error('(%s)unknown Measurement: %s', func_, Measurement);
end
% ----- ChannelType has been discarded

% ----- (MEG/EEG) common procedure
if ~isfield(loadspec, 'ChannelName') || isempty(loadspec.ChannelName)
    loadspec.ChannelName = channel_name;
end

if ~isfield(loadspec, 'ChannelSwitch') || isempty(loadspec.ChannelSwitch)
    loadspec.ChannelSwitch = true;
end

if n_trial > 1
    if ~isfield(loadspec, 'TrialNumber') || isempty(loadspec.TrialNumber)
        % --- all trials
        loadspec.TrialNumber = trial_idx;
    end
    
elseif n_trial == 1
    loadspec.TrialNumber = [];
end

if ~isfield(loadspec, 'Trigger')
    loadspec.Trigger = [];
end

if ~isfield(loadspec, 'Pretrigger')
    loadspec.Pretrigger = [];
end

if ~isfield(loadspec, 'Posttrigger')
    loadspec.Posttrigger = [];
end

if ~isfield(loadspec, 'ActiveChannel') || isempty(loadspec.ActiveChannel)
    if vb_msrmnt_active_channel_is_valid(msrmntfile)
        loadspec.ActiveChannel = true;
    else
        loadspec.ActiveChannel = false;
    end
end

if ~isfield(loadspec, 'ActiveTrial') || isempty(loadspec.ActiveTrial)
    if vb_msrmnt_active_trial_is_valid(msrmntfile)
        loadspec.ActiveTrial = true;
    else
        loadspec.ActiveTrial = false;
    end
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [loadspec, msrmntfile] = inner_check_arguments(loadspec, msrmntfile)
func_ = mfilename;

if isempty(msrmntfile)
    error('(%s)msrmntfile is a required parameter', func_);
end

if exist(msrmntfile, 'file') ~= 2
    error('(%s)cannot find msrmntfile: %s', func_, msrmntfile);
end

if isempty(loadspec) ...
        || ~isfield(loadspec, 'ChannelType')
    loadspec.ChannelType = '';
end

% --- for case insensitive
if ~isempty(loadspec.ChannelType)
    loadspec.ChannelType = upper(loadspec.ChannelType);
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
