function	[ix ,y0, yave] = vb_get_voice_onset(y,level,parm)
% Find voice onset time that smoothed signal exceed the threshold value
%    [ix ,yh, yave] = vb_get_voice_onset(y,level,parm)
% y   : voice signal
% level : threshold level
% parm  : parameter structure
%
% --- Voice onset extraction procedure
% 1. Smoothed amplitude (yave) is calculated by moving average of abs(y) 
%    with time window length of t_smooth (10ms : 100Hz)
% 2. Gamma distribution is fitted to the 'yave' histgram 
% 3. Default threshold value (y0) is determined 
%    from estimated gamma distribution
% 4. threshold value yh = y0 * level
% 5. Voice onsets are extracted where 'yave' exceed the threshold
% 
% --- Condition for voice onset
% 1. Period length that 'yave' exceed the threshold is larger than 't_period'
% 
% --- Optional parameter for voice onset
% parm.t_period : minimum period length that yave exceeds the threshold [50 ms]
% --- Usually following parameters need not be changed
% parm.p_val : P-value corresponding to the threshold  [0.0001]
% parm.t_smooth : moving average window length         [10 ms]
% --- Output
% ix : extracted onset time index
%      if extracted voise length is less than threshold_r, it is rejected
% y0 : amplitude threshold value estimated by using smoothed signal histogram
% yave : smoothed signal by moving average
%      
% 2009-6-11 Masa-aki Sato
% 2011-12-15 Masa-aki Sato
%   Parameters to control onset condition are changed
% 2012-2-22 Masa-aki Sato
%   threshold determination is changed
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Moving average window size
if isfield(parm,'t_smooth'),
	t_smooth = parm.t_smooth;
else
	t_smooth = 10; % 10 ms : 100 Hz
end
% sample length for moving average
tau  = fix( parm.fsamp * t_smooth /1000 ); 

% Moving average with length tau
%yave = filter( ones(tau,1)/tau, 1, abs(y));
yave = filtfilt( ones(tau,1)/tau, 1, abs(y));

if ~isfield(parm,'plot_mode'), parm.plot_mode = []; end;

% Estimate threshold by fitting Gamma distribution to histgram
if ~isfield(parm,'p_val'), parm.p_val = 0.0001; end;

[y0, A ,hy ,ylist] = vb_gamma_dist_param(yave, parm.p_val(1));

if ~isempty(parm.plot_mode) && parm.plot_mode > 0
	vb_show_hist_threshold(y0, A ,hy ,ylist);
end

if ~isempty(level)
	y0 = y0 * level;
end


% extract onset
% find event start (signal exceed the threshold)

if isfield(parm,'t_period')
	t_period  = parm.t_period * parm.fsamp /1000;
else
	t_period  = 100 * parm.fsamp /1000;
end

% Period length that 'yave' exceed 'y0' should be larger than 't_period'
ix = vb_trigger_onset(yave,y0,t_period);

return
