function [meg_info] = vb_ykgwfile_load_extra_channel_info(ykgwfile, meg_info)
% This function reads yokogawa data file and set extra channel information
% to the meg_info
%
% [usage]
%   [meg_info] = vb_ykgwfile_load_extra_channel_info(ykgwfile, meg_info)
%
% [input]
%   ykgwfile : <required> <<file>> yokogawa data file (.con, .raw, .avw)
%   meg_info : <required> <<struct>> MEG information which will be updated
%
% [output]
%   meg_info : <<struct>> updated by adding ExtraChannelInfo struct
%            :  .ExtraChannelinfo <<struct>>
%            :    .Channel_active [N_ex_ch x 1] : active or not
%            :    .Channel_name   [N_ex_ch x 1] : name (label) - string
%            :    .Channel_type   [N_ex_ch x 1] : kind of channel
%            :    .Channel_id     [N_ex_ch x 1] : index number (1 start)
%
% [note]
%   substitute this function for load_yokogawa_eeg_header
%
%   See also
%     vb_yokogawa_get_channel_type
%     getYkgwHdrChannel
%
% [history]
%   2008-02-07 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meg_info', 'var'), meg_info = []; end
if ~exist('ykgwfile', 'var'), ykgwfile = []; end
[meg_info, ExChInfo] = inner_check_arguments(ykgwfile, meg_info);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% get extra channel type
% 2) EXTRA
% 3) REFERENCE
ex_ch_types = vb_yokogawa_get_channel_type(bitor(2,4));
% [N_extra_ch x ]

% load channel information from yokogawa file
channel_info = getYkgwHdrChannel(ykgwfile);

% --- extract extra channels
Channel_active = [];
Channel_name = [];
Channel_type = [];
Channel_id = [];

ch_info_list = channel_info.channel;
ch_type_list = [ch_info_list(:).type];

ch_start = 1;

for i_type = 1:length(ex_ch_types)
  cur_type = ex_ch_types(i_type).id;
  cur_idx = find(ch_type_list(:) == cur_type);
  
  if isempty(cur_idx)
    continue;
  end

  % --- <<data structure>>
  %{
  % ----- EegChannel or EcgChannel
  %   .type
  %   .id
  %   .name : abbreviation name
  %   .gain
  %
  % ----- TriggerChannel or EtcChannel
  %   .type
  %   .id
  %   .name : abbreviation name
  %}

  num_ch = length(cur_idx);

  % active or not (NullChannel or not)
  if cur_type == 0 % NullChannel
    Channel_active = [Channel_active; zeros(num_ch,1)];
  else
    Channel_active = [Channel_active; ones(num_ch,1)];
  end
  Channel_name   = [Channel_name; (cur_idx-1)];
  Channel_type   = [Channel_type; ch_type_list(cur_idx)'];
  Channel_id     = [Channel_id; cur_idx];

  ch_end = ch_start + num_ch - 1;
  ch_start = ch_end + 1;
end

% convert to string - Channel_name
Nch = length(Channel_id);
Channel_name_str = cell(Nch,1);
for i_ch = 1:Nch
  Channel_name_str(i_ch) = {num2str(Channel_name(i_ch))};
end

ExChInfo.Channel_active = Channel_active;
ExChInfo.Channel_name = Channel_name_str;
ExChInfo.Channel_type = Channel_type;
ExChInfo.Channel_id = Channel_id;

meg_info.ExtraChannelInfo = ExChInfo;
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meg_info, ex_info] = inner_check_arguments(ykgwfile, meg_info)

func_ = mfilename;

ex_info = [];

if isempty(meg_info)
  % require no action
else
  if isfield(meg_info, 'ExtraChannelInfo')
    ex_info = meg_info.ExtraChannelInfo;
  end
end

if isempty(ykgwfile)
  error('(%s)ykgwfile is a required parameter', func_);
end

% --- check as a file
if exist(ykgwfile, 'file') ~= 2
  error('(%s)cannot find yokogawa file : %s', func_, ykgwfile);
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
