function [pick, Qpick, CoilWeight, Vcenter, channel_info] = ...
  vb_load_yokogawa_sensor(meg_file, ref_swt, active_swt)
% Load Yokogawa MEG sensor information
% [usage]
%   [pick, Qpick, CoilWeight, Vcenter] = ...
%     vb_load_yokogawa_sensor(meg_file, ref_swt, active_swt)
%
% [input]
%     meg_file : <required> meg-mat file
%      ref_swt : <optional> <<boolean>> flag whether to return normal sensors
%              :  or reference sensors
%              :    true) : reference sensor
%              : [false]) : normal sensor
%   active_swt : <optional> <<boolean>> flag whether to filter active channels
%              :    true) : Yes. This function filters active channels.
%              : [false]) : No. This function does not filter active channels.
%
% [output]
%    pick(1:n,1:3)     : position of detector coils.
%                      : if ref is true, these are positions of refference.
%        (n+1:2n, 1:3) : position of conpensation detector coils.
%    Qpick(1:2n, 1:3)  : direction of coils.
%                      : if ref is true, these are directions of refference
%    CoilWeight        : (m,n) n-th coil weight for m-th sensor channel
%                  basis(channel,dipole) = sensor_weight * basis(coil,dipole)
%    Vcenter           : % origin of sphere model [m]
%    channel_info      : <<struct>> channel information of loaded data
%                        :  .Active [Nchannel x 1]
%                        :  .Name   [Nchannel x 1]
%                        :  .Type   [Nchannel x 1]
%                        :  .ID     [Nchannel x 1]
%                       not supported reference channel info.
% [history]
%   2007-07-10 (Sako) supported refference sensor
%   2009-08-03 (Sako) modified not to use vb_load_meg_info
%   2011-07-21 (Sako) modified how to deal with sensor_weight_ref
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meg_file', 'var'), meg_file = ''; end
if ~exist('ref_swt', 'var'), ref_swt = []; end
if ~exist('active_swt', 'var'), active_swt = []; end
[meg_file, ref_swt, active_swt] = ...
  inner_check_arguments(meg_file, ref_swt, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
channel_info = [];

if ~ref_swt
  % --- meg channels
  load(meg_file, 'MEGinfo', 'pick', 'Qpick');

  CoilWeight = vb_meginfo_get_sensor_weight_meg(MEGinfo);
  
  if isfield(MEGinfo, 'Vcenter')
    Vcenter = MEGinfo.Vcenter;
  else
    Vcenter = [];
  end
  
  % ----- active filter
  ch_info_meg    = vb_load_channel_info(meg_file, 'MEG');
  active_channel = vb_info_get_active_channel(MEGinfo, 1);
  if isempty(active_channel)
    active_swt = false;
  end

  if ~active_swt
    channel_info = ch_info_meg;
    return;
  else
    active_idx = find(active_channel == 1);
    n_channel = vb_info_get_Nchannel(MEGinfo);
    active_coil_idx = [active_idx; (active_idx + n_channel)];

    if ~isempty(pick)
      pick = pick(active_coil_idx, :);
    end
    
    if ~isempty(Qpick)
      Qpick = Qpick(active_coil_idx, :);
    end
    
    if ~isempty(CoilWeight)
      CoilWeight = CoilWeight(active_idx, active_coil_idx);
    end
  end
  channel_info.Active = ch_info_meg.Active(active_idx);
  channel_info.Name   = ch_info_meg.Name(active_idx);
  channel_info.Type   = ch_info_meg.Type(active_idx);
  channel_info.ID     = ch_info_meg.ID(active_idx);
else
  
  % --- reference channels
  ref_pick = [];
  ref_Qpick = [];
  
  load(meg_file, 'MEGinfo', 'ref_pick', 'ref_Qpick');
  
  Nref = size(ref_pick,1);

  % --- sensor_weight_ref
  cur_s_w_r = vb_meginfo_get_sensor_weight_refmg(MEGinfo);
  if isempty(cur_s_w_r)
    MEGinfo = vb_meginfo_init_sensor_weight_refmg(MEGinfo, Nref);
  end
  CoilWeight = vb_meginfo_get_sensor_weight_refmg(MEGinfo);
  
  if isfield(MEGinfo, 'Vcenter')
    Vcenter = MEGinfo.Vcenter;
  else
    Vcenter = [];
  end

  if ~exist('ref_pick', 'var')
    pick = [];
  else
    pick = ref_pick;
  end
  
  if ~exist('ref_Qpick', 'var')
    Qpick = [];
  else
    Qpick = ref_Qpick;
  end

  if ~active_swt
    return;
  end
  
  ch_info_ref = vb_load_channel_info(meg_file, 'REFERENCE');
  if isempty(ch_info_ref)
    active_swt = false;
  end

  % ----- active filter for reference channels
  if ~active_swt
    return;
  else
    active_idx = find(ch_info_ref.Active == 1);
    if ~isempty(pick)
      pick = pick(active_idx, :);
    end
    
    if ~isempty(Qpick)
      Qpick = Qpick(active_idx, :);
    end
    
    if ~isempty(CoilWeight)
      CoilWeight = CoilWeight(active_idx, active_idx);
    end
  end
end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meg_file, ref_swt, active_swt] = ...
  inner_check_arguments(meg_file, ref_swt, active_swt)
func_ = mfilename;

if isempty(meg_file)
  error('(%s) meg_file is a required parameter', func_);
end

if exist(meg_file, 'file') ~= 2
  error('(%s) cannot find meg_file (%s)', func_, meg_file);
end

if isempty(ref_swt)
  ref_swt = false;
end

if isempty(active_swt)
  active_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %

