function [data] = vb_megfile_remove_bias(megfile, procspec)
% remove bias from MEG data
% [usage]
%   [data] = vb_megfile_remove_bias(megfile)
% [input]
%   megfile : <required> <<file>> MEG-MAT file
%   procspec : <optional> <<struct>> specification of removing
%            :  <<fields>>
%            :   .channel_type : channel type
%            :     :  [1]) MEG
%            :     :   2 ) EXTRA
%            :     :   3 ) REFFERENCE
%            :   .method   : how to remove (defined in vb_define_mode)
%            :     :  [0]) REMOVE_BIAS_OFF
%            :     :   1 ) REMOVE_BIAS_PRETRIGGER
%            :     :   2 ) REMOVE_BIAS_LINEAR
% [output]
%      data : processed data
% [note]
%   if specified REMOVE_BIAS_OFF, only returns loaded data
% [history]
%   2007-07-13 (Sako) separated from Yoshioka-san's program
%   2008-06-06 (Sako) modified using new vb_load_meg_data
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%  --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('procspec', 'var'), procspec = []; end
[megfile, procspec] = inner_check_arguments(megfile, procspec);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
const_def = vb_define_mode;
func_ = mfilename;

const = vb_define_const;
switch procspec.channel_type
  case  1  % MEG
    loadspec.ChannelType = const.DATA_TYPE_MAIN;
    data = vb_load_meg_data(megfile, loadspec);
  case  2  % EXTRA
    loadspec.ChannelType = const.DATA_TYPE_EXTRA;
    data = vb_load_meg_data(megfile, loadspec);
  case  3  % REFFERENCE
    loadspec.ChannelType = const.DATA_TYPE_REFERENCE;
    data = vb_load_meg_data(megfile, loadspec);
  otherwise
    error('(%s)unexpected channel_type in procspec : %d', ...
      func_, procspec.channel_type);
end

if procspec.method == const_def.REMOVE_BIAS_OFF
  % do nothing
  return;
end

meginfo = vb_megfile_load_meginfo(megfile);

% every trial - data [Nchannel x Nsample x Ntrial]
for i=1:size(data,3)
  switch procspec.method
    
    case const_def.REMOVE_BIAS_PRETRIGGER
      pre = vb_meginfo_get_pre_trigger(meginfo);

      if ~isempty(pre) && pre > 0, 
        tmp = mean(data(:, 1:pre ,i )')';
        data(:, :, i) = data(:, :, i) - repmat(tmp, [1 size(data, 2)]);
      else
        warning('(%s)there is not pretrigger in MEG-MAT file: %s\n', ...
          func_, megfile);
        % need to confirmation whether it can be returned here or not
        return;
      end
      
    case const_def.REMOVE_BIAS_LINEAR
      % not be implemented yet
  end
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, procspec] = inner_check_arguments(megfile, procspec)
func_ = mfilename;

if isempty(megfile)
  error('(%s)megfile is a required parameter', func_);
end

if exist(megfile, 'file') ~= 2
  error('(%s)cannot find MEG-MAT file : %s', func_, megfile);
end

if isempty(procspec)
  procspec.channel_type = 1;  % MEG
  procspec.method = 0;        % OFF
else
  if  ~isfield(procspec, 'channel_type')
    procspec.channel_type = 1;  % MEG
  end
  if  ~isfield(procspec, 'method')
    procspec.method = 0;        % OFF
  end
end

return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
