function channel_info = vb_megfile_make_channel_info(megfile, ...
  meg_labels, ext_labels, ref_labels)
% make ChannelInfo from channel label
%
% [usage]
%   channel_info = vb_megfile_make_channel_info(megfile, ...
%      meg_labels, ext_labels, ref_labels)
%
% [input]
%      megfile : <required> <<file>> MEG-MAT file
%   meg_labels : <optional> MEG channel label list       {Nchannel_meg x 1}
%   ext_labels : <optional> EXTRA channel label list     {Nchannel_ext x 1}
%   ref_labels : <optional> REFERENCE channel label list {Nchannel_ref x 1}
%
% [output]
%   channel_info : <<struct>>
%                :  .Active [Nchannel x 1]
%                :  .Name   {Nchannel x 1}
%                :  .Type   [Nchannel x 1]
%                :  .ID     [Nchannel x 1]
%                : - Nchannel = Nchannel_meg + Nchannel_ext + Nchannel_ref
%
% [note]
%   See also
%     vb_info_get_channel_info
%
% [history]
%   2008-06-05 (Sako) initial version
%   2009-07-07 (Sako) modified to get data from MEGinfo.ChannelInfo
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = ''; end
if ~exist('meg_labels', 'var'), meg_labels = []; end
if ~exist('ext_labels', 'var'), ext_labels = []; end
if ~exist('ref_labels', 'var'), ref_labels = []; end
[megfile, meg_labels, ext_labels, ref_labels] = ...
  inner_check_arguments(megfile, meg_labels, ext_labels, ref_labels);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
channel_info.Active = [];
channel_info.Name = [];
channel_info.Type = [];
channel_info.ID = [];

meginfo = vb_megfile_load_meginfo(megfile);
org_meg_ch_info = vb_info_get_channel_info(meginfo);
if isempty(org_meg_ch_info)
  warning('(%s) cannot get channel information\n', mfilename);
  return;
end

org_ext_label = vb_meginfo_get_channel_label_extra(meginfo, false, true);

if ~isempty(org_ext_label) && ~isempty(ext_labels)
  [tmp_ext_idx] = vb_util_get_index(org_ext_label, ext_labels);
else
  tmp_ext_idx = [];
end

if ~isempty(org_ext_label) && ~isempty(ref_labels)
  [tmp_ref_idx] = vb_util_get_index(org_ext_label, ref_labels);
else
  tmp_ref_idx = [];
end


if ~isempty(tmp_ext_idx) || ~isempty(tmp_ref_idx)
  if ~isfield(meginfo, 'ExtraChannelInfo') && isempty(meginfo.ExtraChannelInfo)
    warning('(%s)cannot get extra channel information\n', mfilename);
    return;
  else
    % --- correct extra channel information
    extra_info = meginfo.ExtraChannelInfo;

    % --- active of extra channel 
    ext_active = extra_info.Channel_active(tmp_ext_idx);
    ref_active = extra_info.Channel_active(tmp_ref_idx);

    % --- type of extra channel
    ext_type = extra_info.Channel_type(tmp_ext_idx);
    ref_type = extra_info.Channel_type(tmp_ref_idx);

    % --- index of extra channel
    ext_idx = extra_info.Channel_id(tmp_ext_idx);
    ref_idx = extra_info.Channel_id(tmp_ref_idx);
  end
else
  ext_active = [];
  ref_active = [];
  ext_type = [];
  ref_type = [];
  ext_idx = [];
  ref_idx = [];
end

% --- valid MEG channel index
org_meg_label = vb_meginfo_get_channel_label_meg(meginfo, false);
if ~isempty(org_meg_label) && ~isempty(meg_labels)
  [tmp_meg_idx] = vb_util_get_index(org_meg_label, meg_labels);
else
  tmp_meg_idx = [];
end

% ----- Active
org_meg_active = org_meg_ch_info.Active;
meg_active = org_meg_active(tmp_meg_idx);
channel_info.Active = [meg_active;ext_active;ref_active];

% ----- Name
org_meg_labels = org_meg_ch_info.Name;
meg_labels = org_meg_labels(tmp_meg_idx);
channel_info.Name = [meg_labels;ext_labels;ref_labels];

% ----- Type
org_meg_type = org_meg_ch_info.Type;
meg_type = org_meg_type(tmp_meg_idx);
channel_info.Type = [meg_type;ext_type;ref_type];

% ----- ID
org_meg_idx = org_meg_ch_info.ID;
meg_idx = org_meg_idx(tmp_meg_idx);
channel_info.ID = [meg_idx;ext_idx;ref_idx];

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, meg_labels, ext_labels, ref_labels] = ...
  inner_check_arguments(megfile, meg_labels, ext_labels, ref_labels)
func_ = mfilename;
if isempty(megfile)
  error('(%s)megfile is a required parameter', func_);
end

if exist(megfile, 'file') ~= 2
  error('(%s)cannot find megfile : %s', func_, megfile);
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%--- END OF FILE ---%
