function [data, channel_info] = vb_megfile_load_data_external( ...
  megfile, loadspec, new_file, return_swt, verbose_swt)
% load specified data from MEG-MAT file which has data externally (binary)
%
% [usage]
%   [data, channel_info] = vb_megfile_load_data_external( ...
%     megfile, loadspec, new_file, return_swt, verbose_swt)
%
% [input]
%       megfile : <required> <<file>> MEG-MAT file
%
%      loadspec : <required> <<struct>> loading instruction
%      .ChannelName : <optional> channel name list [Nch x 1] []
%                   :  name(string) list of  target channel
%                   :  The way to be used this depends on the
%                   :  "ChannelSwitch" field
%                   :  e.g.
%                   :   {'1';'2';'3';'4'}
%                   :  If this is empty, all the channels are
%                   :  specified
%    .ChannelSwitch : <optiolal> <<boolean>> [true] | false
%                   :   Which is "ChannelName", to be read, or not?
%                   :   [true] : to be read
%                   :    false : to be omitted
%      .ChannelType : <optional> <<string>> channel type
%                   :  [(depend on vb_loadspec_check.m)]
%                   :  'MEG'       : MEG channel data
%                   :  'EXTRA'     : extra channel data
%                   :  'REFERENCE' : reference channel data
%                   :  'AXIAL'     : AxialGradioMeter
%                   :  'PLANAR'    : PlanarGradioMeter
%                   :  'ALL'       : 'MEG'+'EXTRA'+'REFERENCE'
%                   :  - case insensitive
%          .Trigger : <optional> trigger sample number list []
%                   :   [Ntrigger(Ntrial) x 1]
%                   :   e.g. [1000;2000;3000]
%       .Pretrigger : <optional> number of sample of pre-trigger
%                   :  [0]
%      .Posttrigger : <optional> number of sample of post-trigger
%                   :  [Nsample - 1]
%      .TrialNumber : <optional> trial number list [] [Ntrial x 1]
%                   :   "1" start
%                   :    e.g. [1;2;3]
%                   :    if this is empty, all the trials are specified
%    .ActiveChannel : <optional> <<boolean>> [false]
%                   :  active channel filter switch
%                   :    true) active channels
%                   :   false) all the channels
%      .ActiveTrial : <optional> <<boolean>> [false]
%                   :  active trial filter switch
%                   :  valid only when data type is Evoked_Raw
%                   :    true) active trials
%                   :   false) all the trials
%          .saveman : <optional> <<struct>>
%                   :  if you want to store data externally, this field is
%                   :  necessary.
%                   :  if this field is not given or is empty, sampling data
%                   :  will store internally.
%                   :   .data_dir : directory (relative path) for external file
%                   :   .precision : usually 'float64'
%
%      new_file : <optional> new MEG-MAT file name which is stored picked up
%               :  data and changed information []
%               :  if this is empty, new file will not be created
%    return_swt : <optional> <<boolean>> 
%               :  switch whether return loaded data or not
%               :  [true] | false
%   verbose_swt : <optional> <<boolean>> 
%               :  switch whether output verbose message or not
%               :  [true] | false
%
% [output]
%          data : read data [Nchannel x Nsample x Ntrial]
%  channel_info : <<struct>> channel information of loaded data
%               :  .Active [Nchannel x 1]
%               :  .Name   [Nchannel x 1]
%               :  .Type   [Nchannel x 1]
%               :  .ID     [Nchannel x 1]
%
% [note]
%   vb_megfile_load_data was divided into two functions, according to how to
%   store data. internally or externally.
%
%   See also
%     vb_megfile_load_data
%     vb_megfile_load_data_internal
%     vb_megfile_make_new_megfile
%
% [history]
%   2008-04-17 (Sako) initial version
%   2008-04-30 (Sako) deleted unused inner function
%   2008-06-05 (Sako) modified interface spec - ChannelType, channel_info
%   2010-06-30 (Sako) supported storing data externally
%   2010-07-01 (Sako) modified how to pick up trial data
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('loadspec', 'var'), loadspec = []; end
if ~exist('new_file', 'var'), new_file = []; end
if ~exist('return_swt', 'var'), return_swt = []; end
if ~exist('verbose_swt', 'var'), verbose_swt = []; end
[meginfo, ch_list, target_sample, tr_list, ...
  new_file, return_swt, root_dir, loadspec, VERBOSE] = ...
  inner_check_arguments(megfile, loadspec, new_file, return_swt, verbose_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
if VERBOSE >= 5 % temporarily <notice> level
  fprintf('  --- now loading ...\n');
end

%
% --- get channel labels
%
meg_labels = vb_meginfo_get_channel_label_meg(meginfo);
ext_labels = vb_meginfo_get_channel_label_extra(meginfo);
ref_labels = vb_meginfo_get_channel_label_refmg(meginfo);

%
% --- EXTRA CHANNEL DATA
%
if ~isempty(ext_labels)
  [ext_ch_idx] = vb_util_get_index(ext_labels, ch_list);
  if ~isempty(ext_ch_idx)
    ext_ch_idx = vb_util_arrange_list(ext_ch_idx, 0);
  end

else
  ext_ch_idx = [];
end

%
% --- MEG CHANNEL DATA
%
if ~isempty(meg_labels)
  [meg_ch_idx] = vb_util_get_index(meg_labels, ch_list);
  if ~isempty(meg_ch_idx)
    meg_ch_idx = vb_util_arrange_list(meg_ch_idx, 0);
  end
else
  meg_ch_idx = [];
end

%
% --- REFERENCE MAGNETOMETER CHANNEL DATA
%
if ~isempty(ref_labels)
  [ref_ch_idx] = vb_util_get_index(ref_labels, ch_list);
  if ~isempty(ref_ch_idx)
    ref_ch_idx = vb_util_arrange_list(ref_ch_idx, 0);
  end
else
  ref_ch_idx = [];
end

%
% --- get read channels' label
%
read_meg_ch = meg_labels(meg_ch_idx);
read_ext_ch = ext_labels(ext_ch_idx);
read_ref_ch = ref_labels(ref_ch_idx);

% --- MEG
if ~isempty(read_meg_ch)
  bexp = vb_meginfo_read_data_file( ...
    meginfo, read_meg_ch, target_sample, [], tr_list, root_dir);
else
  bexp = [];
end

% --- EXTRA
if ~isempty(read_ext_ch)
    exdata = vb_meginfo_read_data_file( ...
    meginfo, read_ext_ch, target_sample, [], tr_list, root_dir);
else
  exdata = [];
end

% --- REFMG
if ~isempty(read_ref_ch)
  refdata = vb_meginfo_read_data_file( ...
    meginfo, read_ref_ch, target_sample, [], tr_list, root_dir);
else
  refdata = [];
end

% --- actions after loading

% ----- save new meg-mat file
if ~isempty(new_file)
  new_acqtype = vb_meginfo_get_next_acq_type(meginfo, '', loadspec);

  vb_megfile_make_new_megfile(megfile, new_file, loadspec, ...
    read_meg_ch, read_ext_ch, read_ref_ch, ...
    bexp, exdata, refdata, new_acqtype, target_sample);
end


%
% --- return values
%
% ----- arrange data
data = [bexp;exdata;refdata];

% ------- ChannelInfo
%   .Active
%   .Name
%   .Type
%   .ID
channel_info = vb_megfile_make_channel_info(megfile, ...
  read_meg_ch, read_ext_ch, read_ref_ch);

% ----- decide return value
if ~return_swt
  data = [];
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meginfo, read_ch_list, target_samples, trial_idx, ...
    new_file, return_swt, root_dir, loadspec, verbose_level] = ...
  inner_check_arguments(megfile, loadspec, new_file, return_swt, verbose_level)

func_ = mfilename;

if isempty(megfile)
  error('(%s)megfile is a required parameter', func_);
end

if exist(megfile, 'file') ~= 2
  error('(%s)cannot find megfile : %s', func_, megfile);
end

meginfo = vb_megfile_load_meginfo(megfile);

% ----- megfile directory
root_dir = vb_get_file_parts(megfile);

if isempty(loadspec)
  loadspec = vb_loadspec_check(loadspec, eegfile);
end

read_ch_list = loadspec.ChannelName;
% read_ch_list is [N x 1]

% ----- target samples [Ntrial x 2]
target_samples = vb_loadspec_make_trial_sample(loadspec);
if isempty(target_samples)
  n_sample = vb_info_get_sample_number(meginfo);
  target_samples = [1 n_sample];
end

% ----- check TrialNumber
N_trial = vb_info_get_Nrepeat(meginfo);
if N_trial == 1
  trial_idx = [];
else
  org_info = vb_load_measurement_info(megfile);
  org_trial = org_info.Trial;
  org_trial_idx = [org_trial.number];

  % --- trigger info
  if size(target_samples, 1) > 1
    fprintf('(%s) loadspec.Trigger may be wrong\n', mfilename);
    target_samples = target_samples(1,:);
  end
  
  if isempty(loadspec.TrialNumber)
    cur_trial = vb_info_get_trial_data(meginfo);
    trial_idx = 1:length(cur_trial);
  else
    cur_trial = vb_info_get_trial_data(meginfo, loadspec.TrialNumber);
    dst_trial_idx = [cur_trial.number];
    [result, trial_idx] = ...
      vb_util_check_numerical_lists(org_trial_idx, dst_trial_idx);
  end

  if loadspec.ActiveTrial && isequal(org_trial_idx(:), loadspec.TrialNumber(:));
    % loadspec.TrialNumber not specified, by vb_load_meg_data,
    % load only active trial data.
    active_list = vb_info_get_active_trial(meginfo);
    if isempty(active_list)
        active_list = [cur_trial.Active];
    end
    trial_idx = trial_idx(active_list==true);
  end
end
loadspec.TrialNumber = trial_idx;

if isempty(new_file)
  % require no action
end

if isempty(return_swt)
  return_swt = true;
end

if isempty(new_file) && ~return_swt
  error('(%s)wrong combination - empty output file and no return', func_);
end

if isempty(verbose_level)
  verbose_level = 1;
end

return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
