function [ext_data] = vb_megfile_get_internal_data_ext(megfile, ext_type)
% return extra data which is internally stored in megfile
% [usage]
%   [ext_data] = vb_megfile_get_internal_data_ext(megfile, ex_type)
% [input]
%    megfile : <required> <<file>> MEG-MAT file
%   ext_type : <optional> extra type [1]
%            :  1) all the extra channels
%            :  this type depend on MEGinfo.ExtraChannelInfo.Channel_type
%            :  the case of YOKOGAWA data is as follows
%            :    0) NULL CHANNEL
%            :   -1) TRIGGER CHANNEL
%            :   -2) EEG CHANNEL
%            :   -3) ECG CHANNEL
%            :   -4) ETC CHANNEL
% [output]
%   ext_data : extra channel data which is stored in 'bexp_ext' field
%            : [Nchannel x Nsample x Ntrial]
% [note]
%   MEG-MAT file must have bexp_ext and MEGinfo.ExtraChannelInfo fields.
%   This function does not provide data which is selected in sample and trial
% [history]
%   2008-02-15 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('ext_type', 'var'), ext_type = []; end
[megfile, ext_type] = inner_check_arguments(megfile, ext_type);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;

load(megfile, 'bexp_ext');
if ~exist('bexp_ext', 'var')
  warning('(%s)megfile (%s) does not have bexp_ext field\n', func_, megfile);
  bexp_ext = [];
end

% initially set all the data
ext_data = bexp_ext;

if isempty(bexp_ext)
  % return as is
  return;
end

if ext_type == 0
  % return all the channels - not be extracted
  return;
end

meginfo = vb_megfile_load_meginfo(megfile);

% conditional requirement
if ~isfield(meginfo, 'ExtraChannelInfo')
  warning('(%s)meginfo must have ExtraChannelInfo field', func_);
  warning('(%s)return not be extracted data', func_);
  return;
end

% --- normal procedure

ext_idx = vb_meginfo_get_channel_index_extra(meginfo, ext_type);
ext_data = bexp_ext(ext_idx, :, :);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, ext_type] = inner_check_arguments(megfile, ext_type)
func_ = mfilename;

if isempty(megfile)
  error('(%s)megfile is a required parameter', func_);
end

if exist(megfile, 'file') ~= 2
  error('(%s)cannot find megfile : %s', func_, megfile);
end

if isempty(ext_type)
  ext_type = 0;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
