function result = vb_megfile_filter_ch_data(meg_file, proc_spec, new_meg_file)
% make new external channel data files after filtering
% [usage]
%   result = vb_megfile_filter_ch_data(meg_file, proc_spec, new_meg_file)
% [input]
%        meg_file : <required> <<file>> MEG-MAT file (.meg.mat)
%       proc_spec : <required> <<struct>> struct defined process specifications
%                 :  .parm         : <required> <<struct>>
%                 :                :  .filt_suffix
%                 :                :  .bias_flg
%                 :                :  .highpass
%                 :                :  .lowpass
%                 :                :  .highpass_online
%                 :                :  .lowpass_online
%                 :                :  .common_flg
%                 :                :  .fsamp
%                 :  .ch_type      : channel type ['MEG'] - case-insensitive
%                 :  .ch_name      : channel name {Nchannel x 1}
%                 :                : if this is given, it is applied ahead of ch_type
%    new_meg_file : <required> <<file>> filtered MEG-MAT file(.meg.mat)
%                                  : This parameter must include the full extension.
%                                  :  ".meg.mat"
%
% --- Meaning of preprocess parameter
%  .bias_flg    : Bias correction flag  (=0/1/2 : OFF/Bias/Linear)
%               = N > 5: Bias correction by first N sample
%
% --- Filter parameters
%  .highpass    : Highpass filter cutoff frequency [Hz]
%  .lowpass     : Lowpass  filter cutoff frequency [Hz]
%
% --- Type of filter specification
%  .highpass_online : filter order of IIR highpass filter
%      = 0: FIR highpass filter (eegfilt)
%      = 1: online highpass filter (exponential)
%      > 1: Butterworth highpass filter (filtfilt)
%      < 0: Butterworth highpass filter (online)
%
%  .lowpass_online : filter order of IIR lowpass filter
%      = 0: FIR lowpass filter (eegfilt)
%      = 1: online lowpass filter (exponential)
%      > 1: Butterworth lowpass filter (filtfilt)
%      < 0: Butterworth lowpass filter (online)
%
%  .fsamp       : Down sampling frequency [Hz]
%  .common_flg  : Common reference flag (=1/0 : ON/OFF)
%   if these fields are empty, corresponding process is not done
% [output]
%      result : <<struct>> depends on current process
% [note]
%   @see vb_filter_raw_data.m
%   @see vb_megfile_import_extra_data.m
% [history]
%   2009-06-19 (Sako) initial version
%   2009-08-20 (Sako) modified to complain if plural trial data
%   2010-06-25 (Sako) added ch_name and modified help
%   2010-07-02 (Sako) modified progress indicator
%   2016-11-25 (rhayashi) modified specification(new_meg_file)
%   2018-08-23 (rhayashi) Filtering is done by referring to ch_name.
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meg_file', 'var'), meg_file = ''; end
if ~exist('proc_spec', 'var'), proc_spec = []; end
if ~exist('new_meg_file', 'var'), new_meg_file = ''; end
[meg_file, proc_spec, new_meg_file] = inner_check_arguments(meg_file, proc_spec, new_meg_file);

meg_info  = vb_load_measurement_info(meg_file);

% --- this function is for only single trial data
if vb_info_get_Nrepeat(meg_info) > 1
  err_msg = ['this filtering function should be applied to ' ...
    'continuously measured data file without trial extraction.'];
  error('%s', err_msg);
end


% --- MAIN PROCEDURE --------------------------------------------------------- %
%
result = [];

% Preparation for creating new_meg_file
[p_, f_, e_]   = vb_get_file_parts(new_meg_file);
new_meg_dir           = p_;           
new_meg_filename      = [f_, e_];
bin_dir_name          = new_meg_filename(1:strfind(new_meg_filename, '.meg.mat')-1);
proc_spec.new_bin_dir = ['./' bin_dir_name];

% Create binary directory for new_meg_file
new_bin_dir = fullfile(new_meg_dir, proc_spec.new_bin_dir);
if exist(new_bin_dir, 'dir') ~= 7
  vb_mkdir(new_bin_dir);
end

% set Default: ch_type = 'MEG'
if ~isfield(proc_spec, 'ch_name')
    ch_info = vb_load_channel_info(meg_file);
    proc_spec.ch_name = ch_info.Name;
end

% Filtering
inner_filter_ch_data(meg_file, proc_spec, new_meg_file);

fprintf('save: %s\n',new_meg_file);

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meg_file, proc_spec, new_meg_file] = inner_check_arguments(meg_file, proc_spec, new_meg_file)
func_ = mfilename;

% --- check meg_file
if isempty(meg_file)
  error('(%s) meg_file is a required parameter', func_);
end

if exist(meg_file, 'file') ~= 2
  error('(%s) cannot find meg_file : %s', func_, meg_file);
end

if isempty(proc_spec)
  error('(%s) proc_spec is a required parameter', func_);
end

if ~isfield(proc_spec, 'parm')
  error('(%s) parm is a required field of proc_spec', func_);
end
if isempty(new_meg_file)
  error('(%s) new_meg_file is a required parameter', func_);
end
if ~strcmp(new_meg_file(end-7:end), '.meg.mat')
  error('(%s) specified new_meg_file should have ''.meg.mat''', func_);
end

if ~isfield(proc_spec, 'ch_type')
  proc_spec.ch_type = 'MEG';
end
proc_spec.ch_type = upper(proc_spec.ch_type);
return;
%
% --- end of inner_check_arguments()


% --- inner_filter_ch_data()
%
function inner_filter_ch_data(meg_file, proc_spec, new_file)

ch_info = vb_load_channel_info(meg_file, 'ALL');

% filterling channel
filtering_ch_list = proc_spec.ch_name;
if ~iscell(filtering_ch_list)
    filtering_ch_list = {filtering_ch_list};
end

% Original information
meg_info  = vb_load_measurement_info(meg_file);
proc_spec.parm.freq = vb_meginfo_get_sampling_frequency(meg_info);

% Check weather do downsampling or not
do_downsample = false;
if isfield(proc_spec.parm, 'fsamp') && ~isempty(proc_spec.parm.fsamp)
    % Downsampling causes format change.
    do_downsample = true;
    dsamp_parm.freq  = vb_meginfo_get_sampling_frequency(meg_info); % old
    dsamp_parm.fsamp = proc_spec.parm.fsamp; % new
end

% Load data
load_spec.ChannelName   = ch_info.Name;
load_spec.ChannelType   = 'ALL';
load_spec.ActiveChannel = 0; %  all channels
Nall_ch = length(ch_info.Name);

data = vb_load_meg_data(meg_file, load_spec);

% Search filtering target
filter_ch_ix     = [];
for n=1:length(filtering_ch_list)
    ix = strmatch(filtering_ch_list{n}, ch_info.Name, 'exact');
    if isempty(ix)
        error('Unknown channel specified : %s', filtering_ch_list{n});
    else
        filter_ch_ix = [filter_ch_ix; ix];
    end
end

% test filter to get new sampling number.
p_data = vb_filter_raw_data(data(1, :), proc_spec.parm); 
p_data = zeros(Nall_ch, size(p_data, 2));

% Filtering
if do_downsample
    % filtering and downsampling to specified channel
    p_data(filter_ch_ix, :) = vb_filter_raw_data(data(filter_ch_ix, :), proc_spec.parm);

    % apply only downsampling to other channel
    dsamp_ch_ix = setdiff([1:Nall_ch]', filter_ch_ix);
    if ~isempty(dsamp_ch_ix)
        p_data(dsamp_ch_ix, :)  = vb_filter_raw_data(data(dsamp_ch_ix, :), dsamp_parm);
    end
else
    % filtering

    % copy from original
    p_data = data;

    % filtering to specified channel
    p_data(filter_ch_ix, :) = vb_filter_raw_data(data(filter_ch_ix, :), proc_spec.parm);
end

% Create new_file
base_meg = load(meg_file);
base_meg.MEGinfo.saveman.data_dir = proc_spec.new_bin_dir;
base_meg.MEGinfo.Nsample     = size(p_data,2);
base_meg.MEGinfo.SampleFreq  = proc_spec.parm.fsamp;
vb_save_struct(new_file, base_meg);

vb_megfile_import_extra_data(new_file, p_data, ch_info, 1);
return;
%
% --- end of inner_filter_ch_data()
