function	vb_plot_filter_response(ftype,fcut,fs,Norder,fmax,tmax,online)
% plot filter response in frequency and Impulse response
%  vb_plot_filter_response(ftype,fcut,fs,Norder,fmax,tmax,online)
% --- Input (required)
%  ftype ; filter type
%	'low'  : Lowpass  filter
%	'high' : Highpass filter
%	'band' : Bandpass filter
%	'stop' : Stopband filter
%
%  fs   : Sampling frequency [Hz]
%
%  fcut : Cutoff frequency [Hz]
%       = [f1]    Lowpass/Highpass
%       = [f1 f2] Bandpass/Stopband
%
%  Norder = 0: FIR filter (eegfilt)
%  Norder > 0: filter order of Butterworth filter
%
% --- Input (optional)
% fmax : Max frequency in frequency plot [Hz]
%        if it is empty or not given, it is automatically determined
% tmax : Max time in Impulse plot [sec]
%        if it is empty or not given, it is automatically determined
%
% online = 1: online filter with time delay
%        = 0: 'filtfilt' is used for filter with no time delay 
%        if this is not given, 'filtfilt' is used
%
%  2012-9-19 Masa-aki Sato
% 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if nargin<4, 
	help vb_plot_filter_response
	error('input arguments is wrong')
end

Nfreq = 2000;
f  = (0:Nfreq)/Nfreq;
dt = 1/fs;

switch	ftype
case	'low'
	Fmax = min(fcut(1)*3,fs/2);
	Tmax = 10/fcut(1);
case	{'high', 'high_exp'}
	Fmax = min(fcut(1)*5,fs/2);
	Tmax = 10/fcut(1);
case	{'band', 'stop'}
	Fmax = min(fcut(2)*3,fs/2);
	Tmax = 10;
end

if exist('fmax','var')&&~isempty(fmax), Fmax = fmax; end;
if Fmax > fs/2, Fmax = fs/2; end;

t0 = fix(0.5*Tmax/dt);
t  = 0:dt:Tmax;
f  = f*Fmax;
X  = zeros(length(t),1);
X(t0) = 1;
t  = t - t(t0);

if Norder <= 0
	switch	ftype
	case	'low'
		[X ,B] = eegfilt(X', fs, 0, fcut(1) );
	case	'high'
		[X ,B] = eegfilt(X', fs, fcut(1), 0 );
	case	'band'
		[X ,B] = eegfilt(X', fs, fcut(1), fcut(2) );
	case	'stop'
		if Norder==0, Norder = fix(3*(fs/(fcut(2)-fcut(1)))); end;
	    [B, A] = fir1( abs(Norder), fcut./(fs/2), 'stop');
	    X = filtfilt(B, A, X);
	end
	A = 1;
	H = freqz(B,A,f,fs);
elseif strcmp(ftype,'high_exp')==1
	% Exponential online highpass filter
	a = vb_lowpass_init(fs,fcut(1),1);
	A = [1 -a];
	B = 1-a;
	H = 1 - freqz(B,A,f,fs);
	X = vb_online_highpass_cut(X, fs, fcut(1));
else
	switch	ftype
	case	'low'
		[B,A] = butter(Norder, fcut(1)/(fs/2) ,'low');
	case	'high'
		[B,A] = butter(Norder, fcut(1)/(fs/2) ,'high');
	case	'band'
		[B,A] = butter(Norder, fcut(1:2)/(fs/2) );
	case	'stop'
		[B,A] = butter(Norder, fcut(1:2)/(fs/2) , 'stop');
	end
	H = freqz(B,A,f,fs);
	
	if exist('online','var') && online==1
		fprintf('online filter with time delay\n')
		X = filter(B,A, X );
	else
		X = filtfilt(B, A, X );
	end
end	


subplot 221
plot(f,abs(H))
xlabel('Freq[Hz]')
title('Gain of frequency response')
ylim([ 0  1.1])
xlim([f(1) f(end)])
hold on

subplot 222
plot(f,angle(H))
xlabel('Freq[Hz]')
title('Angle of frequency response')
hold on
xlim([f(1) f(end)])

subplot 223
plot(t,X)
xlabel('Time[sec]')
title('Impulse response')
hold on

Xmax = max(X);
plot([0 0],[0 Xmax*1.5],'r--')

if ~exist('tmax','var') || isempty(tmax), 
	ix = find( X > Xmax*0.001 );
	if length(ix) < 100,
		imax = t0+100;
		if imax > length(t)
			tmax = max(t);
		else
			tmax = t(imax);
		end
	else
		tmax = t(ix(end));
	end
end

xlim([-tmax, tmax])

subplot 224
plot(B)
hold on
plot(A,'--')
xlabel('Tap number')
title('Filter coefficient')
xlim([1 max(length(A),length(B))])
fprintf('# of A and B = %d, %d\n',length(A),length(B))
