function	[data] = vb_filter_trial_data(data,parm)
% filtering and down sampling for trial data
%  [data] = vb_filter_trial_data(data,parm)
% --- Input
%  data : input signal [Nchannel x Tsample x Ntrial]
% --- preprocess parameter
% parm
%  .freq        : Sampling frequency [Hz]
%  --- optional fields
%  .bias_flg    : Bias correction flag  
%               = 0/1/2 : OFF/Bias/Linear)
%               = [t1 t2]: Bias correction by time window [t1 t2]
%  .cutfreq     : Cutoff frequency [Hz]
%               = [f1]    for 'low','high'
%               = [f1 f2] for 'band','stop'
%  .fsamp       : Down sampling frequency [Hz]
%                 lowpass or bandpass below Nyquist frequency is necessarily
%  .common_flg  : Common reference flag (= 1/0 : ON/OFF)
% --- If these fields are not given or empty,
%        corresponding process are not applied
% --- Type of filter specification
%  .filter_type : 'low','high','band','stop'
%     if this field is not given or empty
%     filter_type = 'low'  for length(cutfreq) = 1
%     filter_type = 'band' for length(cutfreq) = 2
%  .online : filter order of IIR bandpass filter
%      = 0: FIR filter (eegfilt)
%      > 0: Butterworth filter (filtfilt)
%      < 0: Butterworth filter (online)
%  2009-8-10 Masa-aki Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

[Nch ,T, Ntr] = size(data);

fsamp = parm.freq;

% Bias correction
if  isfield(parm,'bias_flg') && ~isempty(parm.bias_flg) 
	if length(parm.bias_flg) == 2 ,
		% Bias correction by time window [t1 t2]
		%fprintf('Bias correction by first %d sample\n',parm.bias_flg)
		bias = mean(data(:,parm.bias_flg(1):parm.bias_flg(2),:) , 2);
		data = data - repmat(bias, [1 T 1]);
	elseif parm.bias_flg > 0
		% Bias correction by all data
		%fprintf('Bias correction mode = %d\n',parm.bias_flg)
		for m=1:Ntr
			data(:,:,m) = vb_channel_bias_remove(data(:,:,m),parm.bias_flg);
		end
	end
end

% filtering
if isfield(parm,'cutfreq') && ~isempty(parm.cutfreq),
	% cutoff frequency
	fcut  = parm.cutfreq; 
	Nfreq = length(fcut);
	
	if Nfreq > 2, error('Cutoff frequency is wrong'); end;
	
	if isfield(parm,'filter_type') && ~isempty(parm.filter_type),
		filter_type = parm.filter_type;
	else
		if Nfreq==1,
			filter_type = 'low';
		elseif Nfreq==2
			filter_type = 'band';
		end
	end
	
	% Lowpass filter
	if isfield(parm,'online') 
		% fprintf('Online Lowpass filter %6.1f [Hz]\n', fcut)
		% Online Butterworth filter
		Norder = parm.online; % filter order
		if Norder > 0
			%fprintf('butter filtfilt lowpass filter %6.1f [Hz]\n',fcut)
			[B,A] = butter(Norder, fcut/(fsamp/2) ,filter_type);
			
			for m=1:Ntr
				for n=1:Nch
					data(n,:,m) = filtfilt(B,A, data(n,:,m) );
				end
			end
		elseif Norder < 0
			%fprintf('Online butter lowpass filter %6.1f [Hz]\n',fcut)
			[A, B, C, D, Z0] = ...
				vb_online_butter_init(fcut,fsamp,abs(Norder),Nch,filter_type);
			if Nch > 10
				for m=1:Ntr
					Z = Z0;
					[data(:,:,m) ,Z] = ...
					vb_online_filter_loop(A, B, C, D, data(:,:,m), Z);
				end
			else
				for m=1:Ntr
					Z = Z0;
					[data(:,:,m) ,Z] = ...
					vb_online_filter_loop_mex(A, B, C, D, data(:,:,m), Z);
				end
			end
		end
	end
	if ~isfield(parm,'online') || parm.online==0
		% FIR Lowpass filter
		% fprintf('FIR Lowpass filter %6.1f [Hz]\n', fcut)
		switch	filter_type
		case	'low'
			data = eegfilt(data, fsamp, 0, fcut );
		case	'high'
			data = eegfilt(data, fsamp, fcut, 0 );
		case	'band'
			data = eegfilt(data, fsamp, fcut(1), fcut(2) );
		case	'stop'
			Norder = fix(3*(fsamp/(fcut(2)-fcut(1)))); 
		    [B, A] = fir1(Norder, fcut/(fsamp/2), 'stop');
			for m=1:Ntr
				for n=1:Nch
					data(n,:,m) = filtfilt(B,A, data(n,:,m) );
				end
			end
		end
	end
end

% Down sampling
if isfield(parm,'fsamp') && ~isempty(parm.fsamp) && fsamp ~=parm.fsamp
	fsnew = parm.fsamp;
	%fprintf('Down sampling to %6.1f [Hz]\n',fsnew)
	
	data  = vb_convert_freq(data, fsamp, fsnew);
	fsamp = fsnew;
end

% Common reference
if isfield(parm,'common_flg') && ~isempty(parm.common_flg) && parm.common_flg==1
%	fprintf('Common reference\n')
	bias = mean(data,1);
	data = data - repmat(bias, [Nch 1 1]);
end
