function channel_info = vb_eegfile_make_channel_info(eegfile, ch_labels, ch_type)
% make ChannelInfo from channel label
% [usage]
%   channel_info = vb_eegfile_make_channel_info(eegfile, ch_labels, ch_type)
% [input]
%        eegfile : <required> <<file>> EEG-MAT file
%      ch_labels : <optional> specified channel label list {Nchannel x 1}
%        ch_type : <optional> [0] | 1 channel type of channel_info
%                :  0) all (sensor channels and external channel(s)
%                :  1) only sensor channels
%                :  2) only external channel(s)
% [output]
%   channel_info : <<struct>>
%                :  .Active [Nchannel x 1]
%                :  .Name   {Nchannel x 1}
%                :  .Type   [Nchannel x 1]
%                :  .ID     [Nchannel x 1]
%                : - Nchannel = Nchannel_eeg + Nchannel_ext
% [note]
%   for now, main channels and extra channels are managed separately
% [history]
%   2008-06-06 (Sako) initial version
%   2009-07-16 (Sako) added ch_type as the argument
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('eegfile', 'var'), eegfile = ''; end
if ~exist('ch_labels', 'var'), ch_labels = []; end
if ~exist('ch_type', 'var'), ch_type = []; end
[eegfile, ch_labels, ch_type] = ...
  inner_check_arguments(eegfile, ch_labels, ch_type);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
eeginfo = vb_eegfile_load_eeginfo(eegfile);
org_eeg_label = vb_eeginfo_get_channel_label(eeginfo);
org_ext_label = vb_eeginfo_get_channel_label_extra(eeginfo, false);

% ----- EEG CHANNELS
target_eeg_labels = [];
if ch_type ~= 2 && ~isempty(org_eeg_label) && ~isempty(ch_labels)
  [eeg_idx] = vb_util_get_index(org_eeg_label, ch_labels);
  if ~isempty(eeg_idx)
    target_eeg_labels = org_eeg_label(eeg_idx);
  end
else
  eeg_idx = [];
end

% ----- EXTRA CHANNELS
target_ext_labels = [];
if ch_type ~= 1 && ~isempty(org_ext_label) && ~isempty(ch_labels)
  [ext_idx] = vb_util_get_index(org_ext_label, ch_labels);
  if ~isempty(ext_idx)
    target_ext_labels = org_ext_label(ext_idx);
  end
else
  ext_idx = [];
end

extra_info = eeginfo.ExtraChannelInfo;

% ----- Active
if vb_info_active_channel_is_valid(eeginfo)
  active_ch = vb_info_get_active_channel(eeginfo);
  eeg_active = active_ch(eeg_idx);
  
  if ~isempty(extra_info.Channel_active)
    ext_active = extra_info.Channel_active(ext_idx);
  else
    % temporary
    Nch = length(ext_idx);
    ext_active = ones(Nch,1);
  end
  channel_info.Active = [eeg_active;ext_active];
else
  Nch = length(target_eeg_labels)+length(target_ext_labels);
  channel_info.Active = ones(Nch,1);
end

% ----- Name
channel_info.Name = [target_eeg_labels;target_ext_labels];

% ----- Type
% ----- all types are set '1' for now
len_eeg_ch = length(target_eeg_labels);
eeg_type = ones(len_eeg_ch, 1);

if ~isempty(extra_info.Channel_type)
  ext_type = extra_info.Channel_type(ext_idx);
else
%   Nch = length(ext_idx);
%   ext_type = ones(Nch,1);
  % ----- undefined yet
  ext_type = [];
end
channel_info.Type = [eeg_type;ext_type];

% ----- ID
ext_idx = extra_info.Channel_id(ext_idx);
channel_info.ID = [eeg_idx;ext_idx];
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
% function [eegfile, eeg_labels, ext_labels, ref_labels] = ...
%   inner_check_arguments(eegfile, eeg_labels, ext_labels, ref_labels)
function [eegfile, ch_labels, ch_type] = ...
  inner_check_arguments(eegfile, ch_labels, ch_type)

func_ = mfilename;
if isempty(eegfile)
  error('(%s)eegfile is a required parameter', func_);
end

if exist(eegfile, 'file') ~= 2
  error('(%s)cannot find eegfile : %s', func_, eegfile);
end

if isempty(ch_type)
  ch_type = 0; % all channels
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
