function [ template ] = db_template(structureName)
% DB_TEMPLATE: Defines a template structure for all the Brainstorm data types.
%
% USAGE :  [template] = db_template(structureName);
%
% INPUT:   String - {'Anatomy', 'Surface', 'Channel', 'Data', 'HeadModel', 'Results', 'Stat', 
%                    'Subject', 'Study', 'ProtocolInfo', 'ProtocolSubjects', 'ProtocolStudies',
%                    'Layout', 'Colormap', 'Measures', 'LoadedResults', 'FigureId', 
%                    'Figure', 'Dataset', 'DisplayHandlesTimeSeries', 'DisplayHandlesTopography', 
%                    'DisplayHandles3DViz', 'Scout', 'GlobalData', 'Landmark', 
%                    'SCS', 'NCS', 'TessInfo', ...}

% @=============================================================================
% This software is part of the Brainstorm software:
% http://neuroimage.usc.edu/brainstorm
% 
% Copyright (c)2000-2013 Brainstorm by the University of Southern California
% This software is distributed under the terms of the GNU General Public License
% as published by the Free Software Foundation. Further details on the GPL
% license can be found at http://www.gnu.org/copyleft/gpl.html.
% 
% FOR RESEARCH PURPOSES ONLY. THE SOFTWARE IS PROVIDED "AS IS," AND THE
% UNIVERSITY OF SOUTHERN CALIFORNIA AND ITS COLLABORATORS DO NOT MAKE ANY
% WARRANTY, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO WARRANTIES OF
% MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, NOR DO THEY ASSUME ANY
% LIABILITY OR RESPONSIBILITY FOR THE USE OF THIS SOFTWARE.
%
% For more information type "brainstorm license" at command prompt.
% =============================================================================@
%
% Authors: Francois Tadel, 2008-2013


switch lower(structureName)
    % ===== BASIC LOADED STRUCTURES =====
    case 'anatomy'
        template = struct('Comment',  '', ...
                          'FileName', '');
    case 'surface'
        template = struct('Comment',     '', ...
                          'FileName',    '', ...
                          'SurfaceType', '');
    case 'channel'
        template = struct('FileName',        '', ...
                          'Comment',         '', ...
                          'nbChannels',      0, ...
                          'Modalities',      [], ...
                          'DisplayableSensorTypes', []);
    case 'data'
        template = struct('FileName',       '', ...
                          'Comment',        '', ...
                          'DataType',       'recordings', ...
                          'BadTrial',       0);
    case 'dipoles'
        template = struct('FileName',       '', ...
                          'Comment',        '', ...
                          'DataFile',       '');
    case 'headmodel'
        template = struct('FileName',       '', ...
                          'HeadModelType',  '', ...
                          'MEGMethod',      '', ...
                          'EEGMethod',      '', ...
                          'Comment',        '', ...
                          'ECOGMethod',     '', ...
                          'SEEGMethod',     '');
    case 'results'
        template = struct('FileName',       '', ...
                          'Comment',        '', ...
                          'DataFile',       '', ...
                          'isLink',         0, ...
                          'HeadModelType',  '');
    case 'stat'
        template = struct('FileName',       '', ...
                          'Comment',        '', ...
                          'Type',           '', ...
                          'pThreshold',     '', ...
                          'DataFile',       '');
    case 'timefreq'
        template = struct('FileName',       '', ...
                          'Comment',        '', ...
                          'DataFile',       '', ...
                          'DataType',       '');
    case 'image'
        template = struct('FileName',       '', ...
                          'Comment',        '');
    case 'matrix'
        template = struct('FileName',       '', ...
                          'Comment',        '');
    case 'noisecov'
        template = struct('FileName',       '', ...
                          'Comment',        '');
        
    % ==== FILES ====
    case 'mrimat'
        template = struct(...
            'Comment',     '', ...
            'Cube',        [], ...
            'Voxsize',     [], ...
            'NCS',         [], ...
            'SCS',         [], ...
            'Header',      [], ...
            'Histogram',   [], ...
            'InitTransf',  [], ...
            'History',     []);

    case 'surfacemat'
        template = struct(...
            'Comment',         '', ...
            'Vertices',        [], ...
            'Faces',           [], ...
            'VertConn',        [], ...
            'VertNormals',     [], ...
            'Curvature',       [], ...
            'SulciMap',        [], ...
            'Atlas',           db_template('Atlas'), ...
            'iAtlas',          1, ...
            'tess2mri_interp', [], ...
            'Reg',             [], ...
            'History',         []);
        
    case 'datamat'
        template = struct('F',           [], ...
                          'Comment',     '', ...
                          'ChannelFlag', [], ...
                          'Time',        [], ...
                          'DataType',    'recordings', ...
                          'Device',      '', ...
                          'nAvg',        1, ...
                          'Events',      repmat(db_template('event'), 0), ...
                          'History',     []);
    case 'headmodelmat'
        template = struct('MEGMethod',     [], ...
                          'EEGMethod',     [], ...
                          'ECOGMethod',    [], ...
                          'SEEGMethod',    [], ...
                          'Gain',          [], ... 
                          'Comment',       '', ...  [nChannel, 3*nSources]
                          'HeadModelType', '', ...  {'volume', 'surface'}
                          'GridLoc',       [], ...  [nSources, 3]
                          'GridOrient',    [], ...  [nSources, 3]
                          'SurfaceFile',   '', ...
                          'Param',         []);
    case 'resultsmat'
        template = struct('ImagingKernel', [], ...
                          'ImageGridAmp',  [], ...
                          'Whitener',      [], ...
                          'nComponents',   1, ...
                          'Comment',       '', ...
                          'Function',      '', ...
                          'Time',          [], ...
                          'DataFile',      '', ...
                          'HeadModelFile', '', ...
                          'HeadModelType', '', ...
                          'ChannelFlag',   [], ...
                          'GoodChannel',   [], ...
                          'SurfaceFile',   'surface', ...
                          'Atlas',         [], ...
                          'GridLoc',       [], ...
                          'Options',       [], ...
                          'History',       []);
    case 'timefreqmat'
        template = struct('TF',          [], ...
                          'Comment',     '', ...
                          'DataType',    '', ...
                          'Time',        [], ...
                          'TimeBands',   [], ...
                          'Freqs',       [], ...
                          'RefRowNames', [], ...
                          'RowNames',    [], ...
                          'Measure',     [], ...
                          'Method',      [], ...
                          'DataFile',    '', ...
                          'SurfaceFile', [], ...
                          'Atlas',       [], ...
                          'nAvg',        1, ...
                          'Options',     [], ...
                          'History',     []);
    case 'channelmat'
        template = struct(...
            'Comment',    'Channels', ...
            'MegRefCoef', [], ...   % CTF compensators matrix, [nMeg x nMegRef]
            'Projector',  [], ...   % SSP matrix, [nChannels x nChannels]
            'TransfMeg',  [], ...   % MEG sensors: Successive transforms from device coord. system to brainstorm SCS
            'TransfMegLabels', [], ... % Labels for each MEG transformation
            'TransfEeg',  [], ...   % EEG sensors: Successive transforms from device coord. system to brainstorm SCS
            'TransfEegLabels', [], ... % Labels for each EEG transformation
            'HeadPoints', struct(...% Digitized head points 
                'Loc',    [], ...
                'Label',  [], ...
                'Type',   []), ...
            'Channel',    [], ...  % [nChannels] Structure array, one structure per sensor
            'History',     []);
        
    case 'projector'
        template = struct(...
            'Comment',      [], ...
            'Components',   [], ...
            'CompMask',     [], ...
            'Status',       0, ...  % 0: not applied; 1: applied on the fly; 2: saved in the file, not revertible 
            'SingVal',      []);
        
    case 'matrixmat'
        template = struct(...
            'Value',       [], ...
            'Comment',     '', ...
            'Description', [], ...
            'Time',        [], ...
            'ChannelFlag', [], ...
            'nAvg',        1, ...
            'Events',      repmat(db_template('event'), 0), ...
            'History',     []);
    case 'statmat'
        template = struct(...
            'pmap',        [], ...
            'tmap',        [], ...
            'df',          0, ...
            'Type',        '', ...
            'Comment',     '', ...
            'Time',        [], ...
            'ChannelFlag', [], ...
            'HeadModelType', [], ...
            'GridLoc',       [], ...
            'GoodChannel',   [], ...
            'ColormapType',  '', ...
            'Atlas',         [], ...
            'History',     []);
            
    % ==== SUBJECT ====
    case 'subjectmat'
        template = struct(...
            'Comments',    '', ...
            'Anatomy',     '', ...
            'Cortex',      '', ...
            'Scalp',       '', ...
            'InnerSkull',  '', ...
            'OuterSkull',  '', ...
            'UseDefaultAnat',    0, ...
            'UseDefaultChannel', 1); 
    case 'subject'
        template = struct(...
              'Name',                'Unnamed', ...
              'Comments',            '', ...
              'FileName',            '', ...
              'DateOfAcquisition',   '', ...
              'Anatomy',             repmat(db_template('Anatomy'),0), ...
              'Surface',             repmat(db_template('Surface'),0), ...
              'iAnatomy',            [], ...
              'iScalp',              [], ...
              'iCortex',             [], ...
              'iInnerSkull',         [], ...
              'iOuterSkull',         [], ...
              'iOther',              [], ...
              'UseDefaultAnat',      0 , ... 
              'UseDefaultChannel',   1);
  
    % ==== STUDY ====
    case 'studymat'
        template = struct(...
              'DateOfStudy', date, ...
              'Name',        'Unnamed', ...
              'BadTrials',   []);
    case 'study'
         template = struct(...
              'Name',                'Unnamed', ...
              'FileName',            '', ...
              'DateOfStudy',         '', ...
              'BrainStormSubject',   '', ...
              'Condition',           '', ...
              ... repmat(db_template('Channel'),0)
              'Channel',             struct(...  
                  'FileName',        {}, ...
                  'Comment',         {}, ...
                  'nbChannels',      {}, ...
                  'Modalities' ,     {}, ...
                  'DisplayableSensorTypes', {}), ...
              'iChannel',            [], ...
              ... repmat(db_template('Data'),0)
              'Data',              struct(...
                  'FileName',        {}, ...
                  'Comment',         {}, ...
                  'DataType',        {}, ...
                  'BadTrial',        {}), ...
              ... repmat(db_template('HeadModel'),0)
              'HeadModel',         struct(...
                  'FileName',         {}, ...
                  'HeadModelType',    {}, ...
                  'MEGMethod',        {}, ...
                  'EEGMethod',        {}, ...
                  'Comment',          {}, ...
                  'ECOGMethod',       {}, ...
                  'SEEGMethod',       {}), ...
              'iHeadModel',        [], ...
              ... repmat(db_template('Results'),0)
              'Result',              struct(...
                  'FileName',        {}, ...
                  'Comment',         {}, ...
                  'DataFile',        {}, ...
                  'isLink',          {}, ...
                  'HeadModelType',   {}), ...
              ... repmat(db_template('Stat'),  0)
              'Stat',                struct(...
                  'FileName',        {}, ...
                  'Comment',         {}, ...
                  'Type',            {}, ...
                  'pThreshold',      {}, ...
                  'DataFile',        {}), ...
              ... repmat(db_template('Image'),  0)
              'Image',               struct(...
                  'FileName',        {}, ...
                  'Comment',         {}), ...
              ... repmat(db_template('noiseCov'), 0)
              'NoiseCov',            struct(...
                  'FileName',        {}, ...
                  'Comment',         {}), ...
              ... repmat(db_template('Dipoles'), 0)
              'Dipoles',             struct(...
                  'FileName',        {}, ...
                  'Comment',         {}, ...
                  'DataFile',        {}), ...
              ... repmat(db_template('Timefreq'),  0)
              'Timefreq',            struct(...
                  'FileName',        {}, ...
                  'Comment',         {}, ...
                  'DataFile',        {}, ...
                  'DataType',        {}), ...
              ... repmat(db_template('Matrix'),  0)
              'Matrix',             struct(...
                  'FileName',        {}, ...
                  'Comment',         {}));
          
    % ==== PROTOCOLINFO ====
    case 'protocolinfo'
        template = struct(...
              'Comment',               'default_protocol', ...
              'STUDIES',               '', ...
              'SUBJECTS',              '', ...
              'iStudy',                [], ...
              'UseDefaultAnat',        0, ...
              'UseDefaultChannel',     1);     

    case 'protocolsubjects' 
        template = struct(...
            'Subject',        repmat(db_template('Subject'), 0), ...
            'DefaultSubject', []);
        
    case 'protocolstudies'
        template = struct(...
            'Study',         repmat(db_template('Study'), 0), ...
            'DefaultStudy',  db_template('Study'), ...
            'AnalysisStudy', db_template('Study'));
        
    % ==== LAYOUT ====
    case 'layout'
        template = struct(...
            'MainWindowPos',             [0 0 0 0], ...
            'MainWindowSplitHorizontal', 0, ...
            'MainWindowExplorationMode', 'Subjects', ...
            'ScreensDef',                [], ...
            'NoGUI',                     0);     
    
    % ==== EVENT ====
    case 'event'
        template = struct(...
            'label',      '', ...
            'color',      [], ...
            'epochs',     [], ...      % [list of epochs indices]
            'samples',    [], ...      % [list of time indices]
            'times',      [], ...      % [list of time values]
            'reactTimes', [], ...      % [list of reaction times, when applicable]
            'select',     1);
        
    % ==== EPOCH ====
    case 'epoch'
        template = struct(...
                'label',   '', ...        
                'samples', [], ...        % [iStart, iStop]
                'times',   [], ...        % [tStart, tStop]
                'nAvg',    1, ...         % Number of epochs averaged to produce this block
                'select',  1, ...
                'bad',     0, ...
                'channelflag', []);
        
    % ==== FILE IMPORT ====
    case 'sfile'
        template = struct(...
            'filename',   '', ...  % Full path to the file
            'format',     '', ...  % {fif,ctf,egi,neuroscan,brainamp,lena,cartool,...}
            'device',     '', ...  % {neuromag,ctf,egi,neuroscan,brainamp,...}
            'comment',    '', ...
            'byteorder',  '', ...  % ieee-le, ieee-be...
            'prop', struct(...            % Properties of the recordings
                'times',        [], ...   % [tStart, tStop]
                'samples',      [], ...   % [iStart, iStop]
                'sfreq',        0, ...    % Sampling frequency
                'nAvg',         1, ...    % Number of trials used to create this file 
                'currCtfComp',  [], ...   % {0,1,3} Current CTF compensation order already applied to the recordings
                'destCtfComp',  []), ...  % {0,1,3} Destination CTF compensation order
            'epochs',      repmat(db_template('epoch'), 0), ...
            'events',      repmat(db_template('event'), 0), ...
            'header',      [], ...
            'channelmat',  [], ...
            'channelflag', []);
        
    case 'importoptions'
        template = struct(...
            'ImportMode',       'Epoch', ...           % Import mode:  {Epoch, Time, Event}
            'UseEvents',        0, ...                 % {0,1}: If 1, perform epoching around the selected events
            'TimeRange',        [], ...                % Specifying a time window for 'Time' import mode
            'EventsTimeRange',  [-0.1000 0.3000], ...  % Time range for epoching, zero is the event onset (if epoching is enabled)
            'GetAllEpochs',     0, ...                 % {0,1}: Import all arrays, no matter how many they are
            'iEpochs',          1, ...                 % Array of indices of epochs to import (if GetAllEpochs is not enabled)
            'SplitRaw',         0, ...                 % {0,1}: If 1, and if importing continuous recordings (no epoching, no events): split recordings in small time blocks
            'SplitLength',      2, ...                 % Duration of each split time block, in seconds
            'Resample',         0, ...                 % Enable resampling (requires Signal Processing Toolbox)
            'ResampleFreq',     0, ...                 % Resampling frequency (if resampling is enabled)
            'UseCtfComp',       1, ...                 % Get and apply CTF 3rd gradient correction if available 
            'UseSsp',           1, ...                 % Get and apply SSP (Signal Space Projection) vectors if available
            'RemoveBaseline',   'no', ...              % Method used to remove baseline of each channel: {no, all, time, sample}
            'BaselineRange',    [], ...                % [tStart,tStop] If RemoveBaseline is 'time'; Else ignored
            'events',           [], ...                % Events structure: (label, epochs, samples, times, reactTimes, select)
            'CreateConditions', 0, ...                 % {0,1} If 1, create new conditions in Brainstorm database if it is more convenient
            'ChannelReplace',   1, ...        % If 1, prompts for automatic replacement of an existing channel file. If 2, replace it automatically. If 0, do not do it.
            'ChannelAlign',     1, ...        % If 1, prompts for automatic registration. If 2, perform it automatically. If 0, do not do it.
            'IgnoreShortEpochs',1, ...        % If 1, prompts for ignoring the epochs that are shorter that the others. If 2, ignore them automatically. If 0, do not do it.
            'EventsMode',       'ask', ...    % Where to get the events from: {'ask', 'ignore', Filename, ChannelName, ChannelNames}
            'EventsTrackMode',  'ask', ...    % {'value','bit','ask'}   - 4D/BTi only
            'DisplayMessages',  1);           % If 0, do not show any of the message boxes that the user would normally see
        
    % ==== COLORMAPS ====
    case 'colormap'
        template = struct(...
            'Name',             '', ...
            'CMap',             [], ...
            'isAbsoluteValues', 0, ...
            'isNormalized',     0, ...
            'DisplayColorbar',  1, ...
            'MaxValue',         [], ...
            'Contrast',         0, ...
            'Brightness',        0);
        
    case 'colormapinfo'
        template.AllTypes = {};
        template.Type     = '';

    % ==== GLOBAL DATA ====
    case 'measures'
        template = struct(...
            'DataType',        [], ...
            'F',               [], ...
            'Time',            [], ...
            'SamplingRate',    [], ...
            'NumberOfSamples', [], ...
            'ChannelFlag',     [], ...
            'sFile',           [], ...
            'isModified',      0, ...
            'ColormapType',    '');

    case 'loadedresults'
        template = struct(...
            'FileName',         '', ...
            'DataType',         '', ...
            'Comment',          '', ...
            'Time',             [], ...
            'SamplingRate',     [], ...
            'NumberOfSamples',  [], ...
            'ChannelFlag',      [], ...
            'ImageGridAmp',     [], ...
            'ImagingKernel',    [], ...
            'ZScore',           [], ...
            'HeadModelType',    [], ...
            'GridLoc',          [], ...
            'grid2mri_interp',  [], ...
            'nComponents',      0, ...
            'GoodChannel',      [], ...
            'OpticalFlow',      [], ...
            'ColormapType',     '', ...
            'Atlas',            []); 
        
    case 'loadeddipoles'
        template = struct(...
            'FileName',         '', ...
            'Comment',          '', ...
            'Time',             [], ...
            'SamplingRate',     [], ...
            'NumberOfSamples',  [], ...
            'DipoleNames',      [], ...
            'Dipole',           [], ...
            'Subset',           []); 
        
    case 'loadedtimefreq'
        template = struct(...
            'FileName',         '', ...
            'Comment',          '', ...
            'DataFile',         '', ...
            'DataType',         '', ...
            'TF',               [], ...
            'Freqs',            [], ...
            'Time',             [], ...
            'TimeBands',        [], ...
            'SamplingRate',     [], ...
            'NumberOfSamples',  [], ...
            'RefRowNames',      [], ...
            'RowNames',         [], ...
            'Modality',         [], ...
            'Measure',          [], ...
            'Method',           [], ...
            'Options',          [], ...
            'ColormapType',     '', ...
            'Atlas',            [], ...
            'SurfaceFile',      [], ...
            'sPAC',             []); 
        
    case 'loadedmatrix'
        template = struct(...
            'FileName',         '', ...
            'Comment',          '');
        
    case 'loadedsurface'
        template = struct(...
            'FileName',        '', ...
            'Name',            '', ...
            'Comment',         '', ...
            'Vertices',        [], ...
            'Faces',           [], ...
            'VertConn',        [], ...
            'VertNormals',     [], ...
            'VertArea',        [], ...
            'SulciMap',        [], ...
            'tess2mri_interp', [], ...  % Interpolation matrix (Surface -> MRI)
            'mrimask',         [], ...  % MRI mask
            'Atlas',           db_template('Atlas'), ...
            'iAtlas',          1, ...
            'isAtlasModified', 0);
                
     
    case 'loadedmri'
        template = struct(...
            'FileName',   '', ...
            'Comment',    '', ...
            'Cube',       [], ...
            'Voxsize',    [], ...
            'SCS',        [], ...
            'NCS',        [], ...
            'Header',     [], ...
            'Histogram',  [], ...
            'InitTransf', [], ...
            'History',    []);  

    case 'figureid'
        template = struct(...
            'Type',           '', ...
            'SubType',        '', ...
            'Modality',       '');
    
    case 'figure'
        template = struct(...
            'Id',                    db_template('FigureId'), ...   
            'hFigure',               0, ...
            'Handles',               struct(), ...
            'SelectedChannels',      []);

    case 'channeldesc'
        template = struct(...
                'Name',        '', ...
                'Comment',     '', ...
                'Type',        '', ...
                'Loc',         [], ...
                'Orient',      [], ...
                'Weight',      []);

    case 'dataset'
        template = struct(...
            'DataFile',    '', ...
            'StudyFile',   '', ...
            'SubjectFile', '', ...
            'ChannelFile', '', ...
            'Surfaces',    repmat(db_template('LoadedSurface'), 0), ...
            'Measures',    db_template('Measures'), ...
            'Results',     repmat(db_template('LoadedResults'), 0), ...
            'Dipoles',     repmat(db_template('LoadedDipoles'), 0), ...
            'Timefreq',    repmat(db_template('LoadedTimefreq'), 0), ...
            'Matrix',      repmat(db_template('LoadedMatrix'), 0), ...
            'Channel',     repmat(db_template('ChannelDesc'), 0), ...
            'HeadPoints',  [], ...
            'Figure',      repmat(db_template('Figure'), 0));

    case 'displayhandlestimeseries'
        template = struct(...
            'hAxes',         [], ...
            'hCursor',       [], ...
            'hTextCursor',   [], ... 
            'hLines',        [], ...
            'LinesLabels',   [], ...
            'LinesColor',    [], ...
            'hColumnScale',  [], ...
            'hColumnScaleText', [], ...
            'hColumnScaleBar',  [], ...
            'ChannelOffsets',[], ...
            'iDispChannels', [], ...
            'DataMinMax',    [], ...
            'DisplayFactor', [], ...
            'DisplayMode',   []);
    case 'displayhandlestopography'
        template = struct(...
            'hSurf',          [], ...
            'Wmat',           [], ... 
            'DisplayMegGrad', '', ...   % {'first','second','norm'}
            'DataMinMax',     [], ...
            'hSensorMarkers', [], ...
            'hSensorLabels',  [], ...
            'hSensorOrient',  [], ...
            'hContours',      [], ...
            'MarkersLocs',    [], ...
            'hLines',         [], ...  % 2D Layout
            'ChannelOffsets', [], ...  % 2D Layout
            'DisplayFactor',  1, ...   % 2D Layout
            'hCursors',       [], ...  % 2D Layout
            'hZeroLines',     [], ...  % 2D Layout
            'jTextLegend',    [], ...  % 2D Layout
            'Channel',        [], ...  % 2D Layout
            'Vertices',       [], ...  % 2D Layout
            'SelChan',        [], ...  % 2D Layout
            'BoxesCenters',   [], ...  % 2D Layout
            'LinesColor',     []);     % 2D Layout
    case 'displayhandles3dviz'
        template = struct(...
            'hSensorMarkers', [], ...
            'hSensorLabels',  [], ...
            'hSensorOrient',  []);
        
    case 'displayhandlestimefreq'
        template = struct(...
            'DataMinMax', []);
           
    case 'atlas'
        template = struct(...
            'Name',   'User scouts', ...
            'Scouts', repmat(db_template('Scout'), 0));
        
    case 'scout'
        template = struct(...
            'Vertices',    [], ... % Index of vertices
            'Seed',        [], ... % Initial vertex of the scout area
            'Color',       [], ...
            'Label',       '', ...
            'Function',    'Mean', ... % Scout function: PCA, FastPCA, Mean, Mean_norm, Max, Power, All
            'Region',      'UU', ...      % 1st letter: Left/Right/Unknown,  2nd letter: Frontal/Parietal/Temporal/Occipital/Central/Unkown
            'Handles',     repmat(struct( ...
                'hFig',        [], ... % Figure handle in which the scout is displayed
                'hScout',      [], ... % Handles to the graphical scout objects
                'hLabel',      [], ... 
                'hVertices',   [], ...
                'hPatch',      [], ...
                'hContour',    []), 0));
    case 'scoutmat'
        template = struct(...
            'Vertices',   [], ... % Index of vertices
            'Seed',       [], ... % Initial vertex of the scout area
            'Color',      [0 1 0], ...
            'Label',      '', ...
            'Function',   'Mean', ...
            'Region',     'UU');   % 1st letter: Left/Right/Unknown,  2nd letter: Frontal/Parietal/Temporal/Occipital/Central/Unkown
        
    case 'cluster'
        template = struct(...
            'Sensors',    '', ...   % File on which the scout is defined
            'Label',      '', ...   % Comment
            'Function',   'Mean');  % Cluster function: PCA, FastPCA, Mean, Max, Power, All
           
    case 'globaldata'
        template = struct(...
            'Program', struct(...
                'Version',           [], ...
                'BrainstormHomeDir', [], ...
                'GUI',               [], ...
                'ProgressBar',       [], ...
                'Clipboard',         struct(...
                    'Nodes',         [], ...
                    'isCut',         0), ...
                'isReadOnly',        0), ...
            'DataBase', struct(...
                'ProtocolInfo',       [], ...
                'ProtocolSubjects',   [], ...
                'ProtocolStudies',    [], ...
                'isProtocolLoaded',   [], ...
                'isProtocolModified', [], ...
                'iProtocol',          0, ...
                'BrainstormDbDir',    [], ...
                'DbVersion',          0), ...
            'DataBaseSavedTime', 0, ...
            'DataSet',  repmat(db_template('DataSet'), 0), ...
            'Mri',      repmat(db_template('LoadedMri'), 0), ...
            'Surface',  repmat(db_template('LoadedSurface'), 0), ...
            'UserTimeWindow', struct(...
                'Time',            [], ...
                'SamplingRate',    [], ...
                'NumberOfSamples', 0, ...
                'CurrentTime',     []), ...
            'FullTimeWindow', struct(...
                'Epochs', repmat(struct(...
                    'Time',            [], ...
                    'NumberOfSamples', 0), 0), ...
                'CurrentEpoch',    []), ...
            'UserFrequencies', struct(...
                'Freqs',        [], ...
                'iCurrentFreq', []), ...
            'ChannelEditor',  struct(...
                'ChannelFile',    '', ...
                'ChannelMat',     [], ...
                'DataFile',       [], ...
                'LocColumns',     [], ...
                'OrientColumns',  [], ...
                'isModified',     0), ...
            'HeadModeler',    struct(...
                'BFS',            [], ...
                'nbSpheres',      [], ...
                'GUI',            struct(...
                    'hFig',                     [], ...
                    'Figure3DButtonDown_Bak',   [], ...
                    'Figure3DButtonMotion_Bak', [], ...
                    'Figure3DButtonUp_Bak',     [], ...
                    'Figure3DCloseRequest_Bak', [], ...
                    'selectedButton',           [], ...
                    'mouseClicked',             [], ...
                    'isClosing',                [], ...
                    'hButtonTransX',            [], ...
                    'hButtonTransY',            [], ...
                    'hButtonTransZ',            [], ...
                    'hButtonResize',            [])), ...
             'Clusters', repmat(db_template('Cluster'), 0), ...
             'CurrentFigure',        struct(...
                 'Type3D', [], ...
                 'Type2D', [], ...
                 'TypeTF', [], ...
                 'Last',   []), ...
             'CurrentScoutsSurface', '', ...
             'VisualizationFilters', struct(...
                 'LowPassEnabled',       0, ...
                 'LowPassValue',         40, ...
                 'HighPassEnabled',      0, ...
                 'HighPassValue',        1, ...
                 'SinRemovalEnabled',    0, ...
                 'SinRemovalValue',      [], ...
                 'MirrorEnabled',        1), ...
             'Colormaps',            [], ...
             'Preferences',          [], ...
             'ChannelSelections',    struct(...
                 'CurrentSelection', [], ...
                 'Selections',       [], ...
                 'DefaultFactor',    []), ...
             'SelectedRows', [], ...
             'Processes', struct( ...
                 'All',       [], ...
                 'Current',   [], ...
                 'Pipelines', repmat(struct(...
                     'Name',      '', ...
                     'Processes', []), 0), ...
                 'Signature', []), ...
             'ProcessReports', struct(...
                 'jFrameReport', [], ...
                 'jTextHtml',    [], ...
                 'CurrentFile',  [], ...
                 'Reports',      []), ...
             'Interpolations', [], ...
             'Temp',           []);
        template.SelectedRows = {};
        
    case 'landmark'
        template = struct(...
            'Name',   '', ...
            'mriLoc', '');
    case 'scs'
        template = struct(...
            'NAS',    [], ...
            'LPA',    [], ...
            'RPA',    [], ...
            'R',      [], ...
            'T',      []);
    case 'ncs'
        template = struct(...
            'AC',    [], ...
            'PC',    [], ...
            'IH',    []);
        
    % ===== 3DViz appdata structure =====
    case 'tessinfo'
        anatomyColor = [.45*[1 1 1]; .6*[1 1 1]];
        template = struct(...
            'SurfaceFile',                '', ...
            'Name',                    '', ...
            'DataSource',              struct(...
                'Type',                '', ...   % {'data', 'results', 'stats', ...}
                'FileName',            '', ...
                'Atlas',               []), ...
            'ColormapType',            '', ...
            'hPatch',                  [], ...
            'nVertices',               0, ...
            'nFaces',                  0, ...
            'SurfAlpha',               0, ...    % Surface transparency
            'SurfShowSulci',           0, ...    % Toggle show/hide surface sulci map viewing   
            'SurfShowEdges',           0, ...    % Toggle oon/off surface edges display
            'AnatomyColor',            anatomyColor, ... % RGB color for sulci map encoding
            'SurfSmoothValue',         0, ...
            'Data',                    [], ...   % MEG, EEG or current density for overlay to anatomy (needs to be compatible in size with FaceVertexCdata patch property)
            'DataMinMax',              [], ...   % Minimum and maximum of the DataSource.FileName file
            'DataWmat',                [], ...   % Interpolation matrix (transformation to map Data on hPatch surface) 
            'OverlayCube',             [], ...   % Interpolated results in a MRI volume
            'DataAlpha',               0, ...    % Alpha for blending of anatomy and surface data
            'DataThreshold',           0.5, ...  % Threshold to apply to color coding of data values  
            'SizeThreshold',           1, ...    % Threshold to apply to color coding of data values  
            'DataLimitValue',          [], ...   % Relative limits for colormapping
            'CutsPosition',            [0 0 0], ...   % Position of the three orthogonal MRI slices
            'Resect',                  'none', ...    % Either [x,y,z] resect values, or {'left', 'right', 'none'}
            'MipAnatomy',              [], ...        % 3 cells: Maximum intensity power in each direction (MRI amplitudes)
            'MipFunctional',           [] ...         % 3 cellsMaximum intensity power in each direction (sources amplitudes)
            );      
        template.MipAnatomy = cell(3,1);
        template.MipFunctional = cell(3,1);
        
    case 'tfinfo'
        template = struct(...
            'FileName',        '', ...
            'Comment',         '', ...
            'DisplayMode',     '', ...
            'Function',        'power', ...
            'HideEdgeEffects', 0, ...
            'iFreqs',          [], ...
            'RowName',         [], ...
            'InputTarget',     []);
        
    case 'nodelist'
        template = struct(...
           'name',       '', ...
           'comment',    '', ...
           'type',       '', ...
           'jPanel',     [], ...
           'jBorder',    [], ...
           'jTree',      [], ...
           'jPanelTag',  [], ...
           'jCheckTag',  [], ...
           'contents',   [], ...
           'filetags',   [], ...
           'commontags', []);
       
    case 'processfile'
        template = struct(...
            'iStudy',       0, ...
            'iItem',        0, ...
            'FileName',     '', ...
            'FileType',     '', ...
            'Comment',      '', ...
            'Condition',    '', ...
            'SubjectFile',  '', ...
            'SubjectName',  '', ...
            'DataFile',     '', ...
            'ChannelFile',  '', ...
            'ChannelTypes', '');
       
    case 'importfile'
        template = struct(...
            'iStudy',       0, ...
            'iItem',        0, ...
            'FileName',     '', ...
            'FileType',     'import', ...
            'Comment',      'import', ...
            'Condition',    'import', ...
            'SubjectFile',  '', ...
            'SubjectName',  '', ...
            'DataFile',     '', ...
            'ChannelFile',  '', ...
            'ChannelTypes', []);
        
    case 'processdesc'
        template = struct(...
            ... % Process description
            'Function',    [], ...
            'Comment',     '', ...
            'FileTag',     '', ...
            'Description', '', ...
            'Category',    '', ...
            'SubGroup',    '', ...
            'Index',       0, ...
            'isSeparator', 0, ...
            ... % Input type
            'InputTypes',  [], ...
            'OutputTypes', [], ...
            'nInputs',     1, ...
            'nMinFiles',   1, ...
            'isPaired',    0, ...
            'isSourceAbsolute', -1, ...
            'processDim',   [], ...
            'options',      [] ...
            );
        template.InputTypes = {};
        template.OutputTypes = {};
       
    otherwise
        error('Unknown data template : %s', structureName);
end
  
  


