function [result] = vb_posfile_remake_megfile(pos_file, meg_file, new_file)
% remake MEG-MAT file by loading posfile
% [usage]
%   [result] = vb_posfile_remake_megfile(pos_file, meg_file, new_file)
% [input]
%   pos_file : <required> <<file>> POS-MAT file
%   meg_file : <required> <<file>> MEG-MAT file
%   new_file : <optional> new file name if you want [(update meg_file)]
% [output]
%     result : result code
%            :  0) no problem
%            : >1) something wrong
%            :      1) bad pos_file
%            :      2) bad meg_file
% [note]
%   the fields which will be updated are as follows:
%     pick
%     Qpick
%     CoordType
%     PositionFile
%     MEGinfo.MRI_ID
%     MEGinfo.Vcenter
%     MEGinfo.Vradius
%     MEGinfo.device_info
%               .TransInfo
%
% [history]
%   2009-08-05 (Sako) initial version
%   2011-06-01 (Sako) supported standard data format
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('pos_file', 'var'), pos_file = ''; end
if ~exist('meg_file', 'var'), meg_file = ''; end
if ~exist('new_file', 'var'), new_file = ''; end
[pos_file, meg_file, new_file, result] = ...
  inner_check_arguments(pos_file, meg_file, new_file);

if result ~= 0
  return;
end

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
cur_meg = load(meg_file);
% cur_pos = load(pos_file);

% --- cur_pos ------------------------------------------
%            mri_key: 
%     spherical_info: 
%         trans_info: 
%         coord_type: 
%             header: 

meg_info = cur_meg.MEGinfo;

% MRI_ID, Vcenter, Vradius, TransInfo
meg_info = vb_info_add_posfile_info(meg_info, pos_file);

old_CoordType = cur_meg.CoordType;

% pick, Qpick, CoordType
old_pick = cur_meg.pick;
old_Qpick = cur_meg.Qpick;

[new_pick, new_Qpick, new_CoordType] = vb_meg_transform_coordinate( ...
  old_pick, old_Qpick, old_CoordType, pos_file);

cur_meg.pick = new_pick;
cur_meg.Qpick = new_Qpick;
cur_meg.CoordType = new_CoordType;

% ref_pick, ref_Qpick
% --- support standard format
if (isfield(cur_meg, 'ref_pick') && ~isempty(cur_meg.ref_pick)) ...
    && (isfield(cur_meg, 'ref_Qpick') && ~isempty(cur_meg.ref_Qpick))
  old_ref_pick = cur_meg.ref_pick;
  old_ref_Qpick = cur_meg.ref_Qpick;

  [new_ref_pick, new_ref_Qpick] = vb_meg_transform_coordinate( ...
    old_ref_pick, old_ref_Qpick, old_CoordType, pos_file);

  cur_meg.ref_pick = new_ref_pick;
  cur_meg.ref_Qpick = new_ref_Qpick;
end

cur_meg.PositionFile = pos_file;
cur_meg.MEGinfo = meg_info;

vb_save_struct(new_file, cur_meg);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [pos_file, meg_file, new_file, result] = ...
  inner_check_arguments(pos_file, meg_file, new_file)

func_ = mfilename;
result = 0;

if isempty(pos_file)
  fprintf('(%s) pos_file is a required parameter\n', func_);
  result = 1;
  return;
end

if exist(pos_file, 'file') ~= 2
  fprintf('(%s) cannot find pos_file : %s\n', func_, pos_file);
  result = 1;
  return;
end

if isempty(meg_file)
  fprintf('(%s) meg_file is a required parameter\n', func_);
  result = 2;
  return;
end

if exist(meg_file, 'file') ~= 2
  fprintf('(%s) cannot find meg_file : %s\n', func_, meg_file);
  result = 2;
  return;
end

if isempty(new_file)
  new_file = meg_file;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
