function [value, extra_data] = vb_table_query_value(table, key)
% search record by key and return the value of the record.
%
% [USAGE]
%    [value, extra_data] = vb_table_query_value(table, key);
% [IN]
%      key : key for search record. <<string>>
%    table : table data.      [Nx1 struct]
%           .key   : search key <<string>>
%           .value : value. any type is ok.
%           <<extended specification>>
%            - you can add extra fields freely. (e.g. extra1, extra2)
%           .extra1 : any name, any type is ok.
%           .extra2 : any name, any ttype is ok.
% [OUT]
%         value : value in the found record.
%    extra_data : User defined extra fields.
%                 <<struct>> extra fields are returned by structure.
%                  e.g. extra_data.extra1
%                                 .extra2
% [see]
%    vb_key_value_pair.m
%    vb_table_insert.m
%
% [example]
%     tbl = [];
%     % create record
%     key = 'brain_file';
%     value = 'Cortical model file (.brain.mat)';
%     record = vb_key_value_pair(key, value);
%     % insert
%     tbl = vb_table_insert(tbl, record);
%     % inquery
%     value  = vb_table_query_value(tbl, key);
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)


%
% --- Previous check
%
if ~exist('table', 'var')
    error('table is a required parameter.');
end
if ~exist('key', 'var')
    error('key is a required parameter.');
end
if ~ischar(key)
    error('key should be string.');
end
if isempty(table) || ~isstruct(table(1)) || ...
   ~isfield(table(1), 'key') || ~isfield(table(1), 'value')
    error('Unknown format table were specified.');
end

%
% --- Main Procedure
%
value = [];
extra_data = [];

% Search
keys = {table.key};
ix = strmatch(key, keys, 'exact');

if ~isempty(ix)
    if length(ix) >= 2
        error('multiple records were found. Please check table data.');
    end

    % value
    record = table(ix);
    value = record.value;

    % extra_data
    fields = fieldnames(record);
    Nfields = length(fields);
    if Nfields > 2
        % create extra_data
        extra_data = struct;
        for k=1:Nfields
            field_name = fields{k};
            if ~strcmp(field_name, 'key') && ~strcmp(field_name, 'value')
                extra_data.(field_name) = record.(field_name);
            end
        end
    end
end

