function vb_saver_eeg_copy_info(old_file, new_file, new_info)
% (utility to save data) make new EEG-MAT file by copying information from old
% [usage]
%   [new_file] = vb_saver_eeg_copy_info(old_file, new_file, new_info)
%
% [input]
%   old_file : <required> <<file>> EEG-MAT file as a base
%   new_file : <required> EEG-MAT filename with path which will be created
%   new_info : <optional> <<struct>> changing information from old
%            :  if this or following fields are not specified or is empty, 
%            :  value of a parameter will be copied as it is in old_file,
%            :  except for bin_dir field.
%            :  .n_sample      
%            :  .sampling_freq (Hz)
%            :  .pretrigger
%            :  .bin_dir
%            :  -if this is empty or is not specified, File will be deleted.
%
% [output]
%   none
%
% [note]
%
% [history]
%   2013-02-08 (Sako) initial version

% --- CHECK ARGUMENTS --- %
if ~exist('old_file', 'var'), old_file = ''; end
if ~exist('new_file', 'var'), new_file = ''; end
if ~exist('new_info', 'var'), new_info = []; end
[old_file, new_file, new_info] = ...
  inner_check_arguments(old_file, new_file, new_info);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
cur_eeg = load(old_file);
new_eeginfo = cur_eeg.EEGinfo;

% --- update fields
if isempty(new_info)
  if isfield(new_eeginfo, 'File') && isfield(new_eeginfo.File, 'DataDir')
    new_eeginfo.File = rmfield(new_eeginfo.File, 'DataDir');
  end
  
  new_info.n_sample      = vb_eeginfo_get_sample_number(new_eeginfo);
  new_info.sampling_freq = vb_eeginfo_get_sampling_frequency(new_eeginfo);
  new_info.pretrigger    = vb_eeginfo_get_pre_trigger(new_eeginfo);

else
  
  if isempty(new_info.bin_dir)
    if isfield(new_eeginfo, 'File') && isfield(new_eeginfo.File, 'DataDir')
      new_eeginfo.File = rmfield(new_eeginfo.File, 'DataDir');
    end
  else
    new_eeginfo.File.DataDir = new_info.bin_dir;
  end

  if isempty(new_info.n_sample)
    new_info.n_sample = vb_eeginfo_get_sample_number(new_eeginfo);
  end

  if isempty(new_info.sampling_freq)
    new_info.sampling_freq = vb_eeginfo_get_sampling_frequency(new_eeginfo);
  end

  if isempty(new_info.pretrigger)
    new_info.pretrigger = vb_eeginfo_get_pre_trigger(new_eeginfo);
  end
end

% channel info is not changed
new_eeginfo.Nsample         = new_info.n_sample;
new_eeginfo.SampleFrequency = new_info.sampling_freq;
new_eeginfo.Pretrigger      = new_info.pretrigger;
    
cur_eeg.EEGinfo = new_eeginfo;
cur_eeg.eeg_data     = [];

vb_save_struct(new_file, cur_eeg);

return
end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [old_file, new_file, new_info] = ...
  inner_check_arguments(old_file, new_file, new_info)
func_ = mfilename;

% --- old_file
if isempty(old_file)
  error('(%s) old_file is a required parameter', func_);
end

if exist(old_file, 'file') ~= 2
  error('(%s) cannot find old_file : %s', func_, old_file);
end

% --- new_file
if isempty(new_file)
  error('(%s) new_file is a required parameter', func_);
end

% --- new_info
if isempty(new_info)
  % do nothing
end

% ----- new_info.bin_dir
if ~isfield(new_info, 'bin_dir')
  new_info.bin_dir = '';
end

% ----- new_info.n_sample
if ~isfield(new_info, 'n_sample')
  new_info.n_sample = [];
end

% ----- new_info.sampling_freq
if ~isfield(new_info, 'sampling_freq')
  new_info.sampling_freq = [];
end

% ----- new_info.pretrigger
if ~isfield(new_info, 'pretrigger')
  new_info.pretrigger = [];
end

return
end
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
