function vb_remake_megfile_by_bexp(bexp, new_file, org_file, ext_info)
% remake MEG-MAT file in accordance with current bexp data
% [usage]
%   vb_remake_megfile_by_bexp(bexp, new_file, org_file, ext_info)
% [input]
%       bexp : <required> [Nchannel x Nsample x Ntrial]
%   new_file : <required> <<file>> file name which will be remade
%   org_file : <optional> <<file>> original MEG-MAT file
%            :  if this is not empty, some data will be obtained from this file
%   ext_info : <optional> <<struct>>  extension information
%            :  if fields of this struct are specified, they will be overwritten
%            :  to the appropriate fields of MEG-MAT file
%            :  fow now, valid fields are as follows:
%            :   .Pretrigger
%            :   .AcqTypeCode [1|2|3]
%            :     1) 'Evoked_Raw'
%            :     2) 'Evoked_Ave'
%            :     3) 'Continuous_Raw'
%            :   .sampling_freq
% [output]
%   none
% [note]
%   Following parameters are not be able to be decided automatically.
%     pick
%     Qpick
%     MEGinfo.sensor_weight
%     MEGinfo.ChannelInfo.Type
%
% [history]
%   2008-05-09 (Sako) initial version
%   2011-06-02 (Sako) modified according to the new data format
%   2013-02-08 (Sako) added sampling_freq field to ext_info
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('bexp', 'var'), bexp = []; end
if ~exist('new_file', 'var'), new_file = ''; end
if ~exist('org_file', 'var'), org_file = ''; end
if ~exist('ext_info', 'var'), ext_info = []; end
[bexp, new_file, org_file, ext_info] = ...
  inner_check_arguments(bexp, new_file, org_file, ext_info);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
% ----- from bexp
MEGinfo.Nchannel = size(bexp,1);
MEGinfo.Nsample  = size(bexp,2);
MEGinfo.Nrepeat  = size(bexp,3);

for i_tr = 1:MEGinfo.Nrepeat
  MEGinfo.Trial(i_tr).number = i_tr;
  MEGinfo.Trial(i_tr).sample = 1:MEGinfo.Nsample;
  MEGinfo.Trial(i_tr).Active = 1;
end

MEGinfo.MEGch_id = (1:MEGinfo.Nchannel)';
MEGinfo.MEGch_name = cell(MEGinfo.Nchannel,1);
for i_ch = 1:MEGinfo.Nchannel
  MEGinfo.MEGch_name{i_ch} = num2str(MEGinfo.MEGch_id(i_ch));
end

MEGinfo.ActiveChannel = ones(MEGinfo.Nchannel,1);
MEGinfo.ActiveTrial   = ones(MEGinfo.Nrepeat,1);

% ----- else
MEGinfo.Measurement = 'MEG';
MEGinfo.device = 'YOKOGAWA';
MEGinfo.Pretrigger = [];
MEGinfo.SampleFreq = [];
MEGinfo.sensor_weight = [];
MEGinfo.MEG_ID = [];
MEGinfo.MRI_ID = [];
MEGinfo.Vcenter = [];
MEGinfo.Vradius = [];

MEGinfo.ChannelInfo.ID = MEGinfo.MEGch_id;
MEGinfo.ChannelInfo.Name = MEGinfo.MEGch_name;
MEGinfo.ChannelInfo.Active = MEGinfo.ActiveChannel;
MEGinfo.ChannelInfo.Type = [];
MEGinfo.ExtraChannelInfo = [];

MEGinfo.saveman = [];

MEGinfo = vb_info_set_transinfo(MEGinfo, []);
MEGinfo = vb_meginfo_set_acqinfo(MEGinfo, []);
MEGinfo = vb_meginfo_set_acqtype(MEGinfo, '');

% ----- top fields
CoordType = '';
Measurement = '';
pick = [];
Qpick = [];

% ----- from original file
if ~isempty(org_file)
  org_MEGinfo = vb_load_measurement_info(org_file);
  
  if isfield(org_MEGinfo, 'Measurement')
    MEGinfo.Measurement = org_MEGinfo.Measurement;
  end
  if isfield(org_MEGinfo, 'device')
    MEGinfo.device = org_MEGinfo.device;
  end
  if isfield(org_MEGinfo, 'Pretrigger')
    MEGinfo.Pretrigger = org_MEGinfo.Pretrigger;
  end
  if isfield(org_MEGinfo, 'SampleFreq')
    MEGinfo.SampleFreq = org_MEGinfo.SampleFreq;
  end
  if isfield(org_MEGinfo, 'Vcenter')
    MEGinfo.Vcenter = org_MEGinfo.Vcenter;
  end
  if isfield(org_MEGinfo, 'Vradius')
    MEGinfo.Vradius = org_MEGinfo.Vradius;
  end
  if isfield(org_MEGinfo, 'MEG_ID')
    MEGinfo.MEG_ID = org_MEGinfo.MEG_ID;
  end
  if isfield(org_MEGinfo, 'MRI_ID')
    MEGinfo.MRI_ID = org_MEGinfo.MRI_ID;
  end
  if isfield(org_MEGinfo, 'saveman')
    MEGinfo.saveman = org_MEGinfo.saveman;
  end

  % --- device dependent fields
  % ----- device_info.TransInfo
  MEGinfo = ...
    vb_info_set_transinfo(MEGinfo, vb_info_get_transinfo(org_MEGinfo));
  % ----- device_info.acq_info
  MEGinfo = ...
    vb_meginfo_set_acqinfo(MEGinfo, vb_meginfo_get_acqinfo(org_MEGinfo));
  % ----- device_info.acq_type
  acqinfo = vb_meginfo_get_acqinfo(MEGinfo);
  if isfield(acqinfo, 'condition') && isfield(acqinfo.condition, 'type')
    MEGinfo = vb_meginfo_set_acqtype(MEGinfo, acqinfo.condition.type);
  end
  
  % --- coordinate type
  load(org_file, 'CoordType', 'Measurement');
end % if ~isempty(org_file)

% ----- from extension
if ~isempty(ext_info)
  % --- pre-trigger
  if isfield(ext_info, 'Pretrigger')
    MEGinfo.Pretrigger = ext_info.Pretrigger;
  end
  
  % --- data type
  if isfield(ext_info, 'AcqTypeCode')
    switch ext_info.AcqTypeCode
      case 1
        MEGinfo = vb_meginfo_set_acqtype(MEGinfo, 'Evoked_Raw');
      case 2
        MEGinfo = vb_meginfo_set_acqtype(MEGinfo, 'Evoked_Ave');
      case 3
        MEGinfo = vb_meginfo_set_acqtype(MEGinfo, 'Continuous_Raw');
      otherwise
        fprintf('(%s)unknown ext_info.AcqTypeCode : %d\n', ...
          mfilename, ext_info.AcqType);
        % do nothing
    end
  end
  
  % --- sampling frequency
  if isfield(ext_info, 'sampling_freq')
    MEGinfo.SampleFreq = ext_info.sampling_freq;
  end
  
end

if isempty(CoordType)
  CoordType = 'SPM_Right_m';
end

if isempty(Measurement)
  Measurement = 'MEG';
end

% ----- save
vb_fsave(new_file, 'pick', 'Qpick', ...
  'CoordType', 'Measurement', 'bexp', 'MEGinfo');
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [bexp, new_file, org_file, ext_info] = ...
  inner_check_arguments(bexp, new_file, org_file, ext_info)
func_ = mfilename;
if isempty(bexp)
  error('(%s)bexp is a required parameter', func_);
end

if isempty(new_file)
  error('(%s)new_file is a required parameter', func_);
end

if ~isempty(org_file) && exist(org_file, 'file') ~= 2
  error('(%s) cannot find org_file : %s', func_, org_file);
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
