function	[data, channel_info, time_info] = vb_load_meg_data(megfile, load_spec, new_file)
% load measured data of MEG (meg.mat) and EEG (eeg.mat).
%
% This function supports three formats of megfile; Standard MEG format, Minimum MEG format and fileinfo format.
% By default, this function load only active channels and active trials kept in  
% MEGinfo.ActiveChannel and MEGinfo.ActiveTrial in case of Standard MEG format or
% fileinfo.channel_id and fileinfo.act_trial in case of fileinfo format.  
% 
% [Usage]
% 1. Load active channels and trials (Default) 
%    [data] = vb_load_meg_data(megfile);
%
% 2. Load all channels including inactive channels and trials
%    load_spec.ActiveChannel = false;
%    load_spec.ActiveTrial   = false;
%    [data] = vb_load_meg_data(megfile, load_spec);
%
% [input]
%     megfile : <required> <<file>> measurement file
%             :   meg-mat file(.meg.mat)
%             :   eeg-mat file(.eeg.mat)
%
%   load_spec : <optional> <<struct>> loading instruction
%      .ChannelName : <optional> channel name list [Nch x 1] []
%                   :  name(string) list of  target channel
%                   :  The way to be used this depends on the
%                   :  "ChannelSwitch" field
%                   :  e.g.
%                   :   {'1';'2';'3';'4'}
%                   :  If this is empty, all the channels are
%                   :  specified
%    .ChannelSwitch : <optiolal> <<boolean>> [true] | false
%                   :   Which is "ChannelName", to read, or not?
%                   :   [true] : to read
%                   :    false : to omit
%      .ChannelType : <optional> <<string>> channel type ['MEG(EEG)']
%                   :  'MEG'       : MEG channel data (only MEG)
%                   :  'EEG'       : EEG channel data (only EEG)
%                   :  'EXTRA'     : extra channel data
%                   :  'REFERENCE' : reference channel data (only MEG)
%                   :  'AXIAL'     : AxialGradioMeter (only MEG)
%                   :  'PLANAR'    : PlanarGradioMeter (only MEG)
%                   :  'ALL'       : 'MEG(EEG)'+'EXTRA'(+'REFERENCE')
%                   :  'MAG_X'     : Magnetometer OPM channel data toward X-axis (only QZFM device)
%                   :  'MAG_Y'     : Magnetometer OPM channel data toward Y-axis (only QZFM device)
%                   :  'MAG_Z'     : Magnetometer OPM channel data toward Z-axis (only QZFM device)
%                   :  - case insensitive
%          .Trigger : <optional> [Ntrial x 1] [1]
%       .Pretrigger : <optional> [0] equal to zero or over
%      .Posttrigger : <optional> [Nsample - 1] equal to zero or over
%      .TrialNumber : <optional> trial number list [] [Ntrial x 1]
%                   :   "1" start
%                   :    e.g. [1;2;3]
%                   :    if this is empty, all the trials are specified
%    .ActiveChannel : <optional> <<boolean>>
%                   :  active channel filter switch
%                   :   = true  : load active channels [default]
%                   :   = false : load all the channels
%                   :  active information : define MEG(EEG)info.ActiveChannel.
%      .ActiveTrial : <optional> <<boolean>>
%                   :  active trial filter switch
%                   :  valid only when data type is Evoked_Raw
%                   :   = true  : load active trials   [default]
%                   :   = false : load all the trials
%                   :  active information : define MEG(EEG)info.ActiveTrial.
%             (only MEG)
%          .saveman : <optional> <<struct>>
%                   :  When you want to make a new file, loaded data will be
%                   :  stored in external directory.
%                   :  If this field is invalid or switch field is set false,
%                   :  data is stored internally.
%                   :    .data_dir  : data directory (relative path)
%                   :               : ['(name of new_file)_bin']
%                   :               : @see vb_device_make_data_dir.m
%                   :    .precision : data type ['float64']
%                   :    .switch    : <<optional>> switch to store externally
%                   :               :  [true]) store the data externally
%                   :               :   false) store the data internally
%             (only EEG)
%     .bin_data_dir : <optional> relative path from new_file
%                   :  This field is valid only when #3 argument newfile is 
%                   :  specified.
%                   :  if this field is not given or is empty, sampling data
%                   :  will be stored internally.
%
%    new_file : <optional> new MEG-MAT file name which is stored picked up
%             :  data and changed information []
%             :  if this is empty, new file will not be created
%             :  This parameter must include the full extension.
%             :  MEG : ".meg.mat"
%             :  EEG : ".eeg.mat"
%
% [output]
%           data : loaded data [Nchannel x Nsample x Ntrial]
%   channel_info : <<struct>> channel information of loaded data.
%                :  .Active [Nchannel x 1]
%                :  .Name   [Nchannel x 1]
%                :  .Type   [Nchannel x 1]
%                :  .ID     [Nchannel x 1]
%      time_info : <<struct>> time infomation of loaded data.
%                :  .time in sec.          [1x Nsample]
%                :  .sample_frequency(Hz)  [1x1 double]
%
% [note]
%   See also
%     vb_megfile_load_data
%     vb_eegfile_load_data
%     vb_load_channel_info
%
% [Active channel/Active trial information]
%   * Place
%    load(megfile, 'MEGinfo'); % EEG case, MEGinfo -> EEGinfo
%    MEGInfo.ActiveChannel : Active channel list (boolean array) [Nchannelx 1]
%           .ActiveTrial   : Active trial list   (boolean array) [Nrepeat x 1]
%   * Inactivate channel/trial
%     load_spec.ActiveChannel = false;
%     load_spec.ActiveTrial   = false;
%     [data] = vb_load_meg_data(meg_file, load_spec); % [Nchannel x Nsample x Ntrial]
%     MEGinfo.ActiveChannel = ones(size(data, 1), 1); % All channel active
%     MEGinfo.ActiveTrial   = ones(size(data, 3), 1); % All trial   active
%
%     % Inactivate channel/trial
%     MEGinfo.ActiveChannel(N) = 0; % Inactive data(N, :, :)
%     MEGinfo.ActiveTrial(M)   = 0; % Inactive data(:, :, M)
%     then save it. vb_save(megfile, 'MEGinfo');
%
% [history]
%   2006-12-13 (Sako) modified for EEG (acceptable eeg.mat)
%   2007-07-23 (Sako) added loaded channel mode
%   2007-08-10 (Sako) added mode type 'EEG' for load_eeg_data
%   2008-02-20 (Sako) thoroughly revised
%   2008-04-16 (Sako) added .ActiveSwitch in load_spec
%   2008-06-06 (Sako) you can specify ChannelType and get channel_info
%   2008-11-28 (Taku Yoshioka) Use vb_disp() for displaying message
%   2009-07-22 (Sako) added the case only to return internal data
%   2009-08-19 (Sako) modified the case that simply return data,
%                     to return only sensor data (EEG) and channel_info
%   2010-06-30 (Sako) supported storing data externally (only MEG)
%   2010-11-15 (Sako) supported load_spec for 'INFO' file
%   2011-02-08 (Sako) modified for EEG to store data externally
%                     when newfile is valid
%   2011-02-09 (Sako) modified not to output warning message
%                     when cheking internal data
%   2011-06-01 (Sako) supported unknown devices to load
%   2011-07-15 (Sako) modified help about saveman(MEG) and bin_data_dir(EEG)
%   2011-07-15 (Sako) modified the internal EEG data can be loaded correctly
%   2011-08-19 (Sako) modified to avoid errors for the minimum data format
%   2017-04-15 (rhayashi) bugfix:ActiveChannel, ActiveTrial field work correctly.
%   2017-11-24 (rhayashi) return value 'time_info' is added.
%   2022-06-03 (k_suzuki) added case for OPM data
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('load_spec', 'var'), load_spec = ''; end
if ~exist('new_file', 'var'), new_file = ''; end
[megfile, load_spec, new_file] = ...
  inner_check_arguments(megfile, load_spec, new_file);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;

data = [];
channel_info = [];
time_info    = inner_time_info(megfile);

[measurement, device] = vb_load_device(megfile);

Measurement = upper(measurement);
Device      = upper(device);

switch Measurement
  case  'MEG'
    switch Device
      case 'SBI'
        vb_disp(['(' func_ ')return all data because this is ' ...
                 'SBI data file'],VERBOSE_LEVEL_NOTICE);
        [data] = load_sbi_meg_data(megfile);

      case {'YOKOGAWA', 'NEUROMAG'}
        [data, channel_info] = ...
          vb_megfile_load_data(megfile, load_spec, new_file);
              
      case 'QZFM'
        [data, channel_info] = ...
          vb_megfile_load_data(megfile, load_spec, new_file);
      
      case 'BASIC'
        load(megfile, 'bexp');
        if exist('bexp', 'var') && ~isempty(bexp)
          data = bexp;
          channel_info = [];
          return;
        end
      otherwise
        warning('(%s) unknown Device: %s of %s', func_, Device, Measurement);

        % --- simplest mode check
        if isempty(load_spec)
          [result, a_list] = ...
            vb_util_check_variable_in_matfile(megfile, 'bexp');

          if result == a_list.VALID
            load(megfile, 'bexp');
            data = bexp;
            channel_info = [];
            return;
          end
        end

        [data, channel_info] = ...
          vb_megfile_load_data(megfile, load_spec, new_file);
    end
    
  case  'EEG'
    switch Device
      case {'BIOSEMI', 'BRAINAMP'}
        [data, channel_info] = ...
          vb_eegfile_load_data(megfile, load_spec, new_file);

      case 'BASIC'
        load(megfile, 'eeg_data');
        if exist('eeg_data', 'var') && ~isempty(eeg_data)
          data = eeg_data;
          channel_info = [];
          return;
        end

      otherwise
        warning('(%s)unknown Device: %s of %s\n', func_, Device, Measurement);

        % --- simplest mode check
        if isempty(load_spec)
          [result, a_list] = ...
            vb_util_check_variable_in_matfile(megfile, 'eeg_data');
          if result == a_list.VALID
            % --- extract sensor data
            load(megfile, 'eeg_data');
            data = eeg_data;
            channel_info = [];
            return;
          end
        end
        [data, channel_info] = ...
          vb_eegfile_load_data(megfile, load_spec, new_file);
    end

  case  'INFO'
    load(megfile, 'fileinfo');
    
    % --- support loadspec --- %
    if ~isempty(load_spec)
      tmp_file = fileinfo.filename{1};
    
      % ----- check loadspec
      load_spec = vb_loadspec_check(load_spec, tmp_file);

      % ----- narrow channel by user-specified list and active filter
      meginfo = vb_load_meg_info(tmp_file, 4);
      [read_ch, idx] = vb_meginfo_get_read_channels(meginfo, ...
        load_spec.ChannelName, load_spec.ChannelSwitch, ...
        load_spec.ActiveChannel);

      % --- convert loadspec setting to fileinfo setting
      ch_info = vb_info_get_channel_info(meginfo, 2);
      fileinfo.channel_id = ch_info.ID(idx);
    end
    % --- support loadspec --- %
    [data, channel_info] = vb_load_active_trial_data(fileinfo);

  otherwise
    error('(%s)undefined Measurement : %s', Measurement);
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, load_spec, new_file] = ...
  inner_check_arguments(megfile, load_spec, new_file)
func_ = mfilename;

if isempty(megfile)
  error('(%s)megfile is a required parameter', func_);
end

if exist(megfile, 'file') ~= 2
  error('(%s)cannot find megfile : %s', func_, megfile);
end

if isempty(load_spec)
  % require no action
end

if isempty(new_file)
  % require no action
end
return
%
% --- end of inner_check_arguments()

function time_info = inner_time_info(megfile)
time_info = struct;

% Calculate time
meginfo = vb_load_meg_info(megfile);
Nsample = vb_meginfo_get_sample_number(meginfo);
time = vb_index_to_time(1:Nsample, meginfo) / 1000;

% Create time_info
time_info.time = time;
time_info.sample_frequency = vb_meginfo_get_sampling_frequency(meginfo);
%
% --- end of inner_time_info()
    

% --- END OF FILE --- %
