function [Jinfo,Jact,Plist,Jext] = ...
    vb_load_current(currfile,curr_type,ave_mode,trial,ix_area,verbose)
% load estimated current
%
% USAGE:
%  [Jinfo,Jact] = ...
%          vb_load_current(currfile,curr_type,ave_mode,trial,ix_area,verbose)
%
% --- INPUT:
%  currfile  - current file name
% --- Optional INPUT:
% curr_type - current type
%         = 0 : J-current : Cortical current excluding zero activity vertex.
%         = 1 : Z-current : Cortical current on reduced vertex. (bayes_parm_reduce)
%         = 2 : Current   : Cortical current on all vertex.   [Default]
%         = 3 : extra current : Current on extra dipoles (if exist)
%  ave_mode : trial average mode
%          = ON  : trial average is done  [Default]
%          = OFF : No trial average
%  trial   : trial number to load 
%            if empty, all trials are loaded  [Default]
%  ix_area : (absolute) vertex index to load
%            if empty, all vertices are loaded  [Default]
%  verbose : verbose mode for trial file loading
%            0 (No print) , 1 ('.') , 2 ('session , trial')
% --- OUTPUT:
% Jinfo : structure of current info
%    .version   : Version of current file
%    .curr_type : current type(0:J-current, 1:Z-current 2:Current)
%    .tindex    : Subsampled indices
%    .Lact      : # of current direction at one vertex in active region
%    .Wact      : Smoothing filter matrix
%    .ix_act    : Vertex index corresponding to Z-current
%    .ix_act_ex : Vertex index corresponding to J-current
%    .NJact     : # of current vertices
%    .Tsample   : Time sample index for 'Jact'
%    .Ntrial    : # of trials
%    .Nsession  : # of sessions
%    .jactdir   : Current directory
%    .SampleFreq: Sample Frequency [Hz]
%    .Pretrigger: Pretrigger period
%    .Tmsec     : Time in msec
%    .trial     : Trial number to load (optional input for this function)
% 
% Jact    : current timecourse
%   Jact(Njact,Nsample)          for ave_mode = ON 
%   Jact(Njact,Nsample,Ntrials)  for ave_mode = OFF
%     Njact    : # of current
%                   - curr_type = 0: Njact = length(Jinfo.ix_act);
%                   - curr_type = 1: Njact = length(Jinfo.ix_act_ex);
%                   - curr_type = 2: Njact = Nvertex(# of all vertex)
%     Nsample  : # of time sample 
%     Ntrials  : # of trials in all session
%
%   Jact(:, t, :) 
%    Time sample in MEG data : Jinfo.Tsample(t)
%    Time (ms)   in MEG data : Jinfo.Tmsec(t)
%
% Plist : Information of extra dipoles
% Jext  : estimated current for extra dipoles
%
% 2007-3-5 Masa-aki Sato
% 2008-7-9 Masa-aki Sato
%  support all file types and old files
% 2008-09-30 Taku Yoshioka
%  Minor change for variables in Jinfo
% 2010-03-01 M. Sato
%  modified not to use Jinfo.ix_act_ex
% 2022-05-01 k_suzuki
%  Merge with vb_load_current_ext
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)


if ~exist('curr_type','var'), curr_type = 2; end;
if ~exist('ave_mode','var'), ave_mode = ON; end;
if ~exist('trial','var'), trial = []; end;
if ~exist('ix_area','var'), ix_area = []; end;
if ~exist('verbose','var'), verbose = 0; end;

if curr_type~=3
%----- Load cortical current -----
whole_vertex = false;
if curr_type == 2
    whole_vertex = true;
    curr_type = 0;
end
    
[J_type, file_type] = vb_load_current_type(currfile);
%  J_type = 0 : J-current
%  J_type = 1 : Z-current
%  file_type = 0: single file
%  file_type = 1: parent file with trial files

[Jinfo] = vb_load_current_info(currfile);

if ave_mode==OFF && file_type==1
  % trial mode & each trial current file
  if J_type==1
    [Jact,ix_act] = ...
        vb_load_current_z_tr(currfile,curr_type,trial,ix_area,verbose);
    Jinfo.curr_type = curr_type;
  else
    if curr_type==1,
      fprintf('No Z-current and J-current is returned\n');
    end
    [Jact,ix_act] = ...
        vb_load_current_j_tr(currfile,trial,ix_area,verbose);
    Jinfo.curr_type = 0;
  end
else
  % single current file
  if J_type==1
    [Jact,ix_act] = ...
        vb_load_current_z(currfile,curr_type,ave_mode,trial,ix_area);
    Jinfo.curr_type = curr_type;
  else
    if curr_type==1,
      fprintf('No Z-current and J-current is returned\n');
    end
    [Jact,ix_act] = ...
        vb_load_current_j(currfile,ave_mode,trial,ix_area);
    Jinfo.curr_type = 0;
  end
end

Jinfo.trial = trial;
Jinfo.NJact = length(ix_act);

if whole_vertex
    % create cortical current on whole vertex.
    warning('off', 'MATLAB:load:variableNotFound');
    load(currfile, 'Nvertex');
    warning('on',  'MATLAB:load:variableNotFound');
    if exist('Nvertex', 'var')
        [s1, s2, s3] = size(Jact);
        Jact_whole = zeros(Nvertex, s2, s3);
        Jact_whole(Jinfo.ix_act_ex, :, :) = Jact(:, :, :);
        Jact = Jact_whole;
        Jinfo.curr_type = 2;
        Jinfo.NJact     = Nvertex;
    else
        error('Current file format is old. re-run vb_job_current.');
    end
end

else
%----- Load extra current -----
load(currfile,'Pointlist');
if exist('Pointlist', 'var')
    Plist = Pointlist;
    Jext = inner_load_extra_current(currfile, ave_mode, trial);
    Jinfo=[];Jact=[];
else
    warning('Currfile does not contain extra current');
end

end

end

function Jext = inner_load_extra_current(currfile, ave_mode, ix_trial)
const = vb_define_verbose;
VERBOSE_LEVEL_INFO = const.VERBOSE_LEVEL_INFO;

if ave_mode==OFF
  % trial mode
  load(currfile,'Jinfo');
  
  if isfield(Jinfo,'jactdir'),
    jactdir = [fileparts(currfile) '/' Jinfo.jactdir];
    if exist(jactdir,'dir')==0,
      error('No trial data directory')
    end
    
    % Session & Trial number list
    session = [];
    ntrial = [];
    
    for n=1:Jinfo.Nsession
      Ntry = Jinfo.Ntrial(n);
      session = [session, repmat(n,1,Ntry)];
      ntrial  = [ntrial, 1:Ntry];
    end
    
    % Trial loop 
    if isempty(ix_trial), ix_trial = 1:length(ntrial); end
    Ntrial = length(ix_trial);

    for n=1:Ntrial
      nt = ix_trial(n);
      vb_disp(sprintf('Session: %04d , Trial: %04d', ...
                      session(nt), ntrial(nt)), VERBOSE_LEVEL_INFO);
      %if verbose==1,
      %  fprintf('.')
      %elseif verbose==2,
      %  fprintf('Session: %04d , Trial: %04d\n',session(nt), ntrial(nt))
      %end
      
      fname = sprintf('data_s%04dt%04d', session(nt), ntrial(nt));
      load([jactdir '/' fname],'Jext');	
      Jall(:,:,n) = Jext;
    end
    
    Jext = Jall;
  else
    load(currfile,'Jext');
    if ~isempty(ix_trial), Jext = Jext(:,:,ix_trial); end
  end
else
  load(currfile,'Jext');
end
end
    
