function [V0L,F0L,n0L,V0R,F0R,n0R] = vb_load_orig_brain_surf(brain_parm);
% Load BrainVoyager/FreeSurfer files
%   [VL,FL,nL,VR,FR,nR] = vb_load_orig_brain_surf(brain_parm)
%
%--- Input
%
% brain_parm        : structure with following field
%  (used for BrainVoyager files)
%    .BV_left_file   : Brain-Voyager left  surface file
%    .BV_right_file  : Brain-Voyager right surface file
%
%  (used for FreeSurfer files)
%    .FS_left_file   : FreeSurfer left surface file
%    .FS_right_file  : FreeSurfer right surface file
%    .FS_xfm_file    : Talairach transformation file
% 
%--- Output
% VL/R  Left/Right Brain-Voyager (FreeSurfer) vertex coordinate
% FL/R  triangle patch
% nL/R  normal unit vector (Outward)
% 
% --- Coordinate is Right-hand SPM coordinate
% X: Left(-191/2)   -> Right(191/2) 
% Y: Back(-256/2)   -> Front(256/2)
% Z: Bottom(-256/2) -> Top(256/2) 
%
% [history]
% M. Sato  2006-7-21
% 2010-05-28 Taku Yoshioka
%  Minor change (verbose level support)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Define constant
%  (Center of image in BV) = (Center of image in SPM space)
TRANS_COORDINATE = 2;	% = 0 : SBI-compatible-version

TRANS_NORMAL     = 1;

tic; 
if isfield(brain_parm,'BV_left_file') & ...
    ~isempty(brain_parm.BV_left_file),
	%
	% --- Brain-Voyager coordinate
	%
	% [Left-hand coordinate]
	% X:Front(0) -> Back(255)
	% Y:Top(0)   -> Bottom(255)
	% Z:Right(0) -> Left(255)
	
	% V0L/R  Brain-Voyager coordinate
	% n0L/R  inward-normal unit vector 
	% FOL/R  triangle patch

	vb_disp('Read BrainVoyager SRF files ');
	
	Srf = vb_ScriptLoadSRF(brain_parm.BV_left_file);
	V0L = [Srf.VertexX, Srf.VertexY, Srf.VertexZ]; 
	n0L = [Srf.NormalX, Srf.NormalY, Srf.NormalZ]; 
	F0L = Srf.Triangles+1;
	
	Srf = vb_ScriptLoadSRF(brain_parm.BV_right_file);
	V0R = [Srf.VertexX, Srf.VertexY, Srf.VertexZ]; 
	n0R = [Srf.NormalX, Srf.NormalY, Srf.NormalZ]; 
	F0R = Srf.Triangles+1;
	
	vb_disp_nonl('Coordinate transformation (BV to SPM): ');
	
	V0L = vb_bvoyger_to_spm_right(V0L,TRANS_COORDINATE);
	V0R = vb_bvoyger_to_spm_right(V0R,TRANS_COORDINATE);
	n0L = vb_bvoyger_to_spm_right(n0L,TRANS_NORMAL);
	n0R = vb_bvoyger_to_spm_right(n0R,TRANS_NORMAL);
	
	% Outward normal vector
	n0L = -n0L; 
	n0R = -n0R; 

elseif isfield(brain_parm,'FS_left_file') & ...
	    ~isempty(brain_parm.FS_left_file), 

    % Check FS_xfm_file parameter
    if ~isfield(brain_parm,'FS_xfm_file')
       brain_parm.FS_xfm_file = [];
    end

	% FreeSurfer files
	vb_disp_nonl('Read FreeSurfer .asc files: ');
	
	[V0L,F0L] = vb_fs_load_surface(brain_parm.FS_left_file,...
				      brain_parm.FS_xfm_file,1.0);
	[V0R,F0R] = vb_fs_load_surface(brain_parm.FS_right_file,...
				      brain_parm.FS_xfm_file,1.0);
	
	% Outward normal vector
	[F0L,V0L,n0L] = vb_out_normal_surf(F0L,V0L);
	[F0R,V0R,n0R] = vb_out_normal_surf(F0R,V0R);
end

vb_disp(sprintf('%f[sec]',toc));

clear Srf

