function [info, info_type] = vb_load_measurement_info(acq_file, info_type)
% load measurement info data (MEGinfo or EEGinfo)
% [usage]
%   [info, info_type] = vb_load_measurement_info(acq_file, info_type)
% [input]
%    acq_file : <required> measurement file MEG-MAT or EEG-MAT file
%   info_type : <optional> ('EEGINFO' or 'MEGINFO') ['<original type>']
%             :  type of returned information struct (case-insensitive)
%             :  if this is empty, return MEGinfo from MEG-MAT
%             :  and EEGinfo from EEG-MAT
% [output]
%        info : data the format of which is ...
%             :  MEGinfo (info_type == 'MEGinfo') or
%             :  EEGinfo (info_type == 'EEGinfo')
%   info_type : used info_type
% [note]
%   @see inner function : inner_convert_EEGinfo2MEGinfo(EEGinfo)
%      <<MEGinfo>>       <<EEGinfo>>
%     Nchannel      <-- Nchannel
%     Nsample       <-- Nsample
%     Pretrigger    <-- Pretrigger
%     SampleFreq    <-- SampleFrequency
%     Nrepeat       <-- Nrepeat
%     Trial         <-- Trial
%     Vcenter       <-- Vcenter
%     Vradius       <-- Vradius
%     MRI_ID        <-- MRI_ID
%     MEGch_name    <-- ChannelName
%     MEGch_id      <-- ChannelID
%     ActiveChanel  <-- ActiveChannel
%     ActiveTrial   <-- ActiveTrial
%     device        <-- Device
%     Measurement   <-- Measurement
%     ChannelInfo   <-- ChannelInfo
%     ExtraChannelInfo <-- ExtraChannelInfo
%     saveman       <-- File (partially)
% [history]
%   2006-12-13 (Sako) initial version
%   2008-04-15 (Sako) thoroughly rewrote according to new specification
%   2009-08-03 (Sako) added device field when EEG->MEG converting
%   2009-08-10 (Sako) changed how to load MEGinfo, not to use vb_load_meg_info.m
%   2011-05-30 (Sako) modified according to the new data format
%   2011-08-19 (Sako) added checking empty 'File' field before EEG setting
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('acq_file', 'var'), acq_file = ''; end
if ~exist('info_type', 'var'), info_type = ''; end
[acq_file, info_type] = inner_check_arguments(acq_file, info_type);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;
info = [];

% ----- constants
TYPE_EEGINFO = 'EEGINFO';
TYPE_MEGINFO = 'MEGINFO';

[measurement] = vb_load_device(acq_file);
Measurement = upper(measurement);

switch Measurement
  case  'MEG'
    load(acq_file, 'MEGinfo');
    
    if ~isfield(MEGinfo, 'Measurement')
      MEGinfo.Measurement = Measurement;
    end
    info = MEGinfo;

    switch  info_type
      case  TYPE_EEGINFO
        % not be implemented yet
        warning('not be implemented yet (MEGinfo-->EEGinfo)');
        return;
      case  TYPE_MEGINFO
        % nothing to do
        return;
      otherwise
        error('(%s)unknown info_type: %s', func_, info_type);
    end

  case  'EEG'
    load(acq_file, 'EEGinfo');
    if isempty(EEGinfo)
      error('invalid acq_file - no EEGinfo : %s', acq_file);
    end

    switch info_type
      case  TYPE_EEGINFO
        info = EEGinfo;
        return;

      case  TYPE_MEGINFO
        info = inner_convert_EEGinfo2MEGinfo(EEGinfo);
        return;

      otherwise
        error('(%s)unknown info_type: %s', func_, info_type);
    end
    
  case  'INFO'
    load(acq_file, 'fileinfo');
    info = vb_load_measurement_info(fileinfo.filename{1}, info_type);

    Nfiles = length(fileinfo.filename);
    Trial   = [];
    cond_id = [];

    % Initialize Trial information by 1st file.
    if isfield(info, 'Trial')
        Trial = info.Trial;
    end
    % Initialize cond_id by 1st file.
    if isfield(info, 'cond_id')
        cond_id = info.cond_id;
    end
    
    % get Trial information and cond_id from 2nd to last file.
    for k=2:Nfiles
        info_n = vb_load_measurement_info(fileinfo.filename{k}, info_type);
        if isfield(info_n, 'Trial')
            Trial   = [Trial; info_n.Trial];
        end
        if isfield(info_n, 'cond_id')
            cond_id = [cond_id, info_n.cond_id];
        end
    end
    
    % set ActiveChannel, ActiveTrialTrial, and Trial to info structure.
    if isfield(fileinfo, 'ActiveChannel') && ~isempty(fileinfo.ActiveChannel)
        info.ActiveChannel = fileinfo.ActiveChannel;
    end
    if isfield(fileinfo, 'ActiveTrial') && ~isempty(fileinfo.ActiveTrial)
        info.ActiveTrial = fileinfo.ActiveTrial;
        info.Trial = Trial;
    else
        error('fileinfo.ActiveTrial not found.');
    end
    
    % The number of Trials, cond_id is set to info structure.
    if isfield(fileinfo, 'Ntotal')
        info.Nrepeat = fileinfo.Ntotal;
    else
        info.Nrepeat = length(info.ActiveTrial);
    end
    info.cond_id = cond_id;
    
    otherwise
    error('unknown Measurement : %s - maybe invalid measurement_file: %s', ...
      Measurement, measurement_file);
end

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_convert_EEGinfo2MEGinfo()
%
%   returned fields of MEGinfo : 
%     Nchannel         <-- Nchannel
%     Nsample          <-- Nsample
%     Pretrigger       <-- Pretrigger
%     SampleFreq       <-- SampleFrequency
%     Nrepeat          <-- Nrepeat
%     Trial            <-- Trial
%     Vcenter          <-- Vcenter
%     Vradius          <-- Vradius
%     MRI_ID           <-- MRI_ID
%     MEGch_name       <-- ChannelName
%     MEGch_id         <-- ChannelID
%     ActiveChannel    <-- ActiveChannel
%     ActiveTrial      <-- ActiveTrial
%     device           <-- Device
%     Measurement      <-- Measurement
%     ChannelInfo      <-- ChannelInfo
%     ExtraChannelInfo <-- ExtraChannelInfo
%     (saveman)        <-- File & DataType
function MEGinfo = inner_convert_EEGinfo2MEGinfo(EEGinfo)
  if ~isfield(EEGinfo, 'Nchannel'),        EEGinfo.Nchannel = []; end
  if ~isfield(EEGinfo, 'Nsample'),         EEGinfo.Nsample = []; end
  if ~isfield(EEGinfo, 'Pretrigger'),      EEGinfo.Pretrigger = []; end
  if ~isfield(EEGinfo, 'SampleFrequency'), EEGinfo.SampleFrequency = []; end
  if ~isfield(EEGinfo, 'Nrepeat'),         EEGinfo.Nrepeat = []; end
  if ~isfield(EEGinfo, 'Trial'),           EEGinfo.Trial = []; end
  if ~isfield(EEGinfo, 'Vcenter'),         EEGinfo.Vcenter = []; end
  if ~isfield(EEGinfo, 'Vradius'),         EEGinfo.Vradius = []; end
  if ~isfield(EEGinfo, 'ChannelName'),     EEGinfo.ChannelName = []; end
  if ~isfield(EEGinfo, 'ChannelID'),       EEGinfo.ChannelID = []; end
  if ~isfield(EEGinfo, 'ActiveChannel'),   EEGinfo.ActiveChannel = []; end
  if ~isfield(EEGinfo, 'ActiveTrial'),     EEGinfo.ActiveTrial = []; end
  if ~isfield(EEGinfo, 'Device'),          EEGinfo.Device = []; end
  if ~isfield(EEGinfo, 'Measurement'),     EEGinfo.Measurement = []; end
  if ~isfield(EEGinfo, 'MRI_ID'),          EEGinfo.MRI_ID = []; end
  if ~isfield(EEGinfo, 'ChannelInfo'),     EEGinfo.ChannelInfo = []; end
  if ~isfield(EEGinfo, 'ExtraChannelInfo'),EEGinfo.ExtraChannelInfo = []; end
  if ~isfield(EEGinfo, 'File'),            EEGinfo.File = []; end
  if ~isfield(EEGinfo, 'DataType'),        EEGinfo.DataType = []; end

  MEGinfo.Nchannel      = EEGinfo.Nchannel;
  MEGinfo.Nsample       = EEGinfo.Nsample;
  MEGinfo.Pretrigger    = EEGinfo.Pretrigger;
  MEGinfo.SampleFreq    = EEGinfo.SampleFrequency;
  MEGinfo.Nrepeat       = EEGinfo.Nrepeat;
  MEGinfo.Trial         = EEGinfo.Trial;
  if isfield(EEGinfo, 'cond_id')
      MEGinfo.cond_id   = EEGinfo.cond_id;
  end
  MEGinfo.Vcenter       = EEGinfo.Vcenter;
  MEGinfo.Vradius       = EEGinfo.Vradius;
  MEGinfo.MEGch_name    = EEGinfo.ChannelName;
  MEGinfo.MEGch_id      = EEGinfo.ChannelID;
  MEGinfo.ActiveChannel = EEGinfo.ActiveChannel;
  MEGinfo.ActiveTrial   = EEGinfo.ActiveTrial;
  MEGinfo.device        = EEGinfo.Device;
  MEGinfo.Measurement   = EEGinfo.Measurement;
  MEGinfo.MRI_ID        = EEGinfo.MRI_ID;
  MEGinfo.ChannelInfo   = EEGinfo.ChannelInfo;
  MEGinfo.ExtraChannelInfo = EEGinfo.ExtraChannelInfo;
  if ~isempty(EEGinfo.File)
    MEGinfo.saveman.data_dir = ...
      [EEGinfo.File.OutputDir '/' EEGinfo.File.DataDir];
  else
    MEGinfo.saveman.data_dir = '';
  end
  
  if ~isempty(EEGinfo.DataType)
    MEGinfo.saveman.precision = EEGinfo.DataType(1);
  else
    MEGinfo.saveman.precision = '';
  end
  return;
%
% --- end of inner_convert_EEGinfo2MEGinfo()

% --- inner_check_arguments()
%
function [acq_file, info_type] = inner_check_arguments(acq_file, info_type)
func_ = mfilename;
if isempty(acq_file)
  error('(%s)acq_file is a required parameter', func_);
end

if isempty(info_type)
  [measurement] = vb_load_device(acq_file);
  Measurement = upper(measurement);
  
  switch Measurement
    case 'MEG'
      info_type = 'MEGINFO';
    case 'EEG'
      info_type = 'EEGINFO';
    case  'INFO'
      load(acq_file, 'fileinfo');
      [Measurement2] = vb_load_device(fileinfo.filename{1});
	  switch Measurement2
	    case 'MEG'
	      info_type = 'MEGINFO';
	    case 'EEG'
	      info_type = 'EEGINFO';
	  end
    otherwise
      error('(%s)unexpected Measurement : %s', func_, Measurement);
  end
else
  info_type = upper(info_type);
end
return;
%
% --- end of inner_check_arguments()

% --- END OF FILE --- %
