function [V,F] = vb_fs_load_surface(ascfile,xfmfile,reduce_ratio)
% Load surface data created by FreeSurfer and 'mris_convert' 
% (.asc file). Vertex number is reduced. Inverse talairach 
% transformation is applied. 
%
% --- Syntax
% function [V,F] = vb_fs_load_surface(ascfile,xfmfile,reduce_ratio)
%
% --- Input
% ascfile: include cortical structure created by FreeSurfer
% reduce_ratio: Reduction ratio of vertex number (0-1)
% xfmfile: Linear transformation file (Talairach transformation)
%
% - This program supports linear transformation only (.xfm). 
%
% --- History
% 2005-08-24 Taku Yoshioka
% 2008-10-16 Taku Yoshioka
%   Input variable 'xfmfile' can be empty. 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Load surface data
[F0,V0] = load_surface(ascfile); % inner function

% Vertex reduction
if nargin>=3 & reduce_ratio<1, 
  [F,V] = vb_reducepatch(F0,V0,reduce_ratio); 
else 
  F = F0; V = V0; 
end

if nargin>=2 & ~isempty(xfmfile), 
  % Load .xfm file
  xfm = load_xfm(xfmfile);

  % Inverse transformation
  xfm_inv = inv(xfm); 
  Vext = [V'; ones(1,size(V,1))];
  V = (xfm_inv*Vext)'; 
  V = V(:,1:3); 
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Inner functions
%
function [F,V] = load_surface(ascfile)

% Open file
fid = fopen(ascfile,'r');
fgetl(fid); % comment line
tmp = fscanf(fid,'%d %d',2); % # of vertices and patches
NV = tmp(1);
NF = tmp(2);
tmp = fscanf(fid,'%e %e %e %e',inf);
V = reshape(tmp(1:NV*4),[4 NV])';
F = reshape(tmp(NV*4+1:end),[4 NF])';
V = V(:,1:3); 
F = F(:,1:3)+1;

fclose(fid);

function xfm = load_xfm(xfmfile)

% Open file
fid = fopen(xfmfile,'r');

% Search line starts with 'Linear_Transform'
chk = 0;
while chk==0, 
  str = fgetl(fid);
  if strcmp(str,'Linear_Transform ='), chk = 1; end; 
end

% Read three lines (transformation matrix)
xfm = zeros(4,4); 
for i=1:3, xfm(i,:) = str2num(fgetl(fid)); end;
xfm(4,:) = [0 0 0 1];

fclose(fid); 

