function vb_job_convert_spm
% Save SPM result as '.spm.mat' file format
% (VBMEG public function)
%
% % This function convert thresholded statistical values, obtained by
% SPM2/5/8 software, with its coordinate and save them as
% .spm.mat file, which is used in VBMEG to incorporate fMRI
% information in MEG current source estimation. 
%
% [syntax]
% vb_job_convert_spm
%
% Note that you have to execute this function on the MATLAB console 
% SPM2/5/8 is running. When you display statistical map of some kind of
% contrast, their value and its coordinate are stored in MATLAB
% workspace. This function just save these data in an appropriate
% format. 
%
% [output]
% .spm.mat file is created. 
%
% [history]
% Originally written by Naokazu Goda (for SPM99)
% 2004-12-11 Taku Yoshioka
% 2004-12-14 Naokazu Goda (backward coordinate transformation routine)
% 2005-02-23 Dai Kawawaki 
% 2008-09-17 Dai Kawawaki
% 2008-10-27 Taku Yoshioka
% 2010-05-12 Takeda added spm8 support
% 2010-06-22 Taku Yoshioka
%  Use 'spm_get_orig_coord.m' for SPM5/8
% 2010-10-06 Masa-aki Sato
%  Use 'spm_get_orig_coord_spm5.m' for SPM5/8
% 2016-03-28 Masa-aki Sato
%  Whole program is rewritten
% 2016-05-31 rhayashi
%   - SPM result(MNI) is saved as MNI coordinate value.
%   - SPM result(Subject) : is saved as SPM_Right_mm.
%                         : NIfTI format is supported.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%%%%%% DEBUG %%%%%%
DEBUG = 0;

%
% VBMEG constant
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
coord_const = vb_define_coordinate;

%
% Get SPM result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if DEBUG==0
%%%% If DEBUG =1, SPM check is skipped %%%%%%%%
    try
      spm_ver = spm('ver');
    catch
      error('You can use this function on result display of SPM analysis.');
    end
    xSPM = get_important_struct(spm_ver);
else
    xSPM = [];
end

savefile  = [];

%
% Select Standard brain or Subject brain
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
mesg = {'Which coordinate is used in the current SPM result?'};
title_txt = 'SPM data import';

ansA = 'Normalised template brain(MNI)';
ansB = 'Subject brain';

STD_sw = questdlg(mesg, title_txt, ansA, ansB, ansA);
switch STD_sw
case ansA
    xSPM.coord_type = coord_const.COORDINATE_MNI_RIGHT_MM;
case ansB
    % Select Image file to adjust position gap of fmri activity derived from image center position.
    imagefile = get_t1_image;
    if isempty(imagefile)
        disp('T1 image file was not selected.');
        return;
    end
    fprintf('Transform subject native to SPM-right-mm coordinate : ');
    fprintf('imagefile %s\n',imagefile);

    %
    % Transform from subject's native to SPM-right-mm coordinate
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if DEBUG==0
        %%%% If DEBUG =1, this part is skipped %%%%%%%%
        xSPM = vb_nifti_mm_to_spm_right_mm(xSPM', imagefile)';
        xSPM.coord_type = coord_const.COORDINATE_SPM_RIGHT_MM;
    end %%%% DEBUG END

otherwise
    error('Unknown coordinate is specified.');
end

%
% Get Save file name
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[dir,fnames] = vb_file_select({'.spm.mat'},'Input Save filename(.spm.mat)',true);
if isempty(fnames), 
  disp('.spm.mat file was not selected.');
  return;
end
savefile = [dir filesep fnames{1}];

%
% Save
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fprintf('Save file: %s\n',savefile)

if DEBUG==1, return; end
    
SPM = xSPM;
vb_fsave(savefile,'SPM', 'spm_ver');

return;

%
% Inner function
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function xSPM = get_important_struct(spm_ver)
switch (lower(spm_ver))
 case 'spm2' 
  disp('Check SPM version: SPM2');
  
  command = 'exist(''xSPM'')';
  if ~evalin('base',command) % execute command in base workspace
    error('Some contrast image must be displayed in SPM.');
  else
    mSPM = evalin('base','xSPM');
    nSPM.Z = mSPM.Z;
    nSPM.XYZmm = mSPM.XYZmm;
    nSPM.STAT = mSPM.STAT;
    xSPM = nSPM;
    clear mSPM;
  end
  clear command;

 case 'spm5'
  disp('Check SPM version: SPM5');

  command = 'exist(''xSPM'')';
  if ~evalin('base',command) % execute command in base workspace
    error('Some contrast image must be displayed in SPM.');
  else
    mSPM = evalin('base','xSPM');
    nSPM.Z = mSPM.Z;
    nSPM.XYZmm = mSPM.XYZmm;
    nSPM.STAT = mSPM.STAT;
    xSPM = nSPM;
    clear mSPM;
  end
  clear command;

  case 'spm8'
   disp('Check SPM version: SPM8');

   command = 'exist(''xSPM'')';
   if ~evalin('base',command) % execute command in base workspace
     error('Some contrast image must be displayed in SPM.');
   else
     mSPM = evalin('base','xSPM');
     nSPM.Z = mSPM.Z;
     nSPM.XYZmm = mSPM.XYZmm;
     nSPM.STAT = mSPM.STAT;
     xSPM = nSPM;
     clear mSPM;
   end
   clear command;
   
 otherwise
  error('Sorry, this function does not accommodate to your spm version.');
end

function imagefile = get_t1_image
imagefile = '';
[dir,fnames] = vb_file_select({'.hdr','.nii'},'Input T1 image file(.hdr/.nii)');
if ~isempty(fnames)
    imagefile = fullfile(dir, fnames{1});
end

function xSPM = backword_transform_normalized_to_subject_native(snfile, spm_ver, xSPM)
switch (lower(spm_ver))
 case 'spm2', 
  sn = load(snfile);
  xSPM.XYZmm=vb_unsn(xSPM.XYZmm,sn);
  
 case {'spm5','spm8'}, 
  xSPM.XYZmm = spm_get_orig_coord_spm5((xSPM.XYZmm)',snfile);
  xSPM.XYZmm = (xSPM.XYZmm)';
end

