function [VBfilt, Jinfo, bayes_parm, vb_parm] ...
          = vbmeg_current_filter_z_meeg(varargin)
% Calculate Hierarchical Bayesian inverse filter for MEG+EEG
% (VBMEG public function)
% The scale of inverse filter (KW) is original digit. (not division by bsnorm_meeg)
% Usage: J = VBfilt.KW*[B; E];  %B and E are MEG and EEG data.
%
% [syntax]
% [VBfilt, Jinfo, bayes_parm, vb_parm] ...
%    = vbmeg_current_filter_z(filt_parm);
% [VBfilt, Jinfo, bayes_parm, vb_parm] ...
%    = vbmeg_current_filter_z(proj_root, filt_parm); [old style]
%
% [input]
% proj_root: <<string>> VBMEG project root directory. 
% filt_parm: <<struct>> Parameters for calculation of inverse filter. 
% --- fields of filt_parm
%  bayesfile : <<string>> Current variance file (.bayes.mat). 
%  filterfile: <optional> <<string>> inverse filter file (.vbfilt.mat). If
%              this field is not defined, inverse filter file is not
%              saved. 
%  basisfile_meg : <optional> <<string>> If this field is given, leadfield
%              file of this field is used for calculation of inverse filter.
%  basisfile_eeg : <optional> <<string>> same as above.
%  extra.basisfile : <optional> <<string>> If this field is given,
%              leadfield for extra dipole is given by this field
%  ix_area   : <optional> <<vector>> Vertex indices for which inverse
%              filter is calculated (Z-current). 
% ---
%
% [output]
% VBfilt: <<struct>> Inverse filter and related variables.
% --- fields of VBfilt
%  sx    : Observation noise variance (Nwindow,Ntask).
%  Cov   : Normalized sensor noise covariance matrix {Ntask}(Nch,Nch).
%  KW    : VB inverse filter  (Njact,Nch,Nwindow,Ntask).
%  KW_ext: VB inverse filter for extra-dipole.
%  ix_act: Vertex index corresponding to active current (Njact,1).
% ---
%
% Jinfo : <<struct>> Information of cortical current. 
% --- fields of Jinfo
%  Lact     : <<int>> Number of current direction.
%  ix_act   : <<vector>> Vertex index corresponding to Z-current.
%  ix_act_ex: <<vector>> Vertex index corresponding to J-current.
%  Wact     : <<matrix>> Spatial smoothing matrix of focal window.
% ---
%
% [note]
% J-current is calculated using Wact and Z-current
%    Jact = Wact * Zact
% 
% 2008-10-29 M. Sato
% 2010-05-27 Taku Yoshioka
%  Extra-dipole support
% 2010-12-06 taku-y
%  [trivial] Jinfo.version = vb_version. 
% 2010-12-07 taku-y
%  [trivial] Jinfo.version = vbmeg('version');
%
% 2017-03-13 Yusuke Fujiwara
%  MEG+EEG
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if length(varargin) == 1
  proj_root = [];
  filt_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  filt_parm = varargin{2};
end

vb_disp('--- Bayesian inverse filter calculation ');

% multiple bayse file for multiple conditions
bayesfile = filt_parm.bayesfile;

if ~iscell(bayesfile), bayesfile = {bayesfile}; end

Ntask = length(bayesfile);

if ~isempty(proj_root)
  for n=1:Ntask
    bayesfile{n} = fullfile(proj_root, bayesfile{n});
  end
end  

if Ntask > 1
  vb_disp(['Multiple condition filters (N=' num2str(Ntask) ...
           ') are calculated']);
end

%%%%%% load VBMEG estimated result
load(bayesfile{1}, 'bayes_parm','Model','vb_parm','Model_ext');

%%%%%% check parameter of 'filt_parm'
% Values of 'filt_parm' fields dominates over 
%   those of 'bayes_parm' in bayesfile
[bayes_parm, ix_area] = check_arg(bayes_parm, filt_parm);
			
if ~isempty(proj_root)
  bayes_parm_abs = vb_parm_absolute_path(proj_root, bayes_parm);
else
  bayes_parm_abs = bayes_parm;
end

%%%%%% Preparation of lead fields
% Gact   : leadfield of focal window
% ix_act : Vertex index corresponding to active current Zact
% ix_act_ex : Vertex index corresponding to active current Jact
% Wact   : Spatial smoothing matrix of focal window

% Focal window
vb_disp('--- Preparation of lead field matrix ');

%MEG leadfield 
lf_parm.basisfile = [proj_root filesep bayes_parm_abs.basisfile_meg];

lf_parm.brainfile = bayes_parm_abs.brainfile;
lf_parm.areafile = bayes_parm_abs.areafile;
lf_parm.patch_norm = bayes_parm_abs.patch_norm;
lf_parm.expand_spatial_filter = bayes_parm_abs.expand_spatial_filter;
lf_parm.spatial_smoother = bayes_parm_abs.spatial_smoother;
lf_parm.area_key = bayes_parm_abs.area_key;
lf_parm.reduce = bayes_parm_abs.reduce;
lf_parm.Rfilt = bayes_parm_abs.Rfilt;
lf_parm.remove_area_key = [];

[Gact_b, ix_act, ix_act_ex, Wact, Lact]  =  ...
	 vb_leadfield_preparation(lf_parm);

 
%EEG leadfield 
lf_parm.basisfile = [proj_root filesep bayes_parm_abs.basisfile_eeg]; 
Gact_e = vb_leadfield_preparation(lf_parm);

for n = 1:length(Gact_b)
    Gact{n} = [Gact_b{n}./Model.bsnorm_meg; Gact_e{n}./Model.bsnorm_eeg];
    Gact0{n} = [Gact_b{n}; Gact_e{n}];
end


%%%%%% Area index in which current is calculated
if ~isempty(ix_area)
  % Select vertex index 'ix_area' within the active current region
  [jx_area_ex, ix_area_ex] = vb_index2indexsub(ix_area, ix_act_ex);
else
  jx_area_ex = 1:length(ix_act_ex);
end

Wact   = Wact(jx_area_ex,:);
jx_act = find( sum(Wact, 1) > 0);
Wact   = Wact(:,jx_act);

% active index of Z-current
ix_act = ix_act(jx_act);
% active index of J-current
ix_act_ex = ix_act_ex(jx_area_ex);

% # of active vertex
Njact_area = length(jx_act);

%% Constant
Nsession = length(Gact);	 % Number of sessions

if Nsession > 1, error('Multiple leadfield is not supported');end

G = Gact{1};    % Ga 
Gt = G';

[Nj, Nwindow] = size(Model.a);
Nch = size(Model.Cov{1},1);

% Extra-dipole support
if ~isempty(Model_ext)
  vb_disp('Lead field matrix of extra-dipole ');
  Get = vb_load_basis(bayes_parm_abs.extra.basisfile{1});
  Ge = Get';
end

%%%%%% Estimated current variance
a_inv = zeros(Nj, Nwindow, Ntask);
sx    = zeros(Nwindow, Ntask);
Covs  = cell(1,Ntask);

for n=1:Ntask
  load(bayesfile{n}, 'Model');
  a_inv(:,:,n) = Model.a;
  sx(:,n)    = Model.sx;
  Covs{n}  = Model.Cov{1};
  
  % Extra-dipole support
  if ~isempty(Model_ext)
    e_inv(:,:,n) = Model_ext.a;
  end
end

Hj = zeros(Nwindow,Ntask);
KW = zeros(Njact_area,Nch,Nwindow,Ntask);
SB_cov = zeros(Nch,Nch,Nwindow,Ntask);

% Debug code
%S2.a = a_inv(:,1);
%S2.G = G;
%S2.Covs = Covs; 
%save('tmp2.mat','S2');
%error('aho');

if ~isempty(Model_ext)
  %%%%%% Time window loop
  for j=1:Nwindow
    for n=1:Ntask
      Cov = Covs{n};     % Sg
    
      %%%% Calculate Inverse filter for current reconstruction
      A_z = a_inv(:,j,n);
      A_e = e_inv(:,j,n);
		
      % Inverse filter for each session
      % SB  = Ga * a_inv *Ga' + Cov      
      SB      = G * vb_repmultiply(Gt, A_z) + Cov;  % Sigma_B
      SB      = SB + Ge * vb_repmultiply(Get, A_e);      
      SB_inv  = pinv( SB );        % Sigam_B^{-1}
      SB_cov(:,:,j,n) = SB_inv;

      Hj(j,n)  = - 0.5*( vb_log_det(SB) + Nch*log(sx(j,n)) );
        
      %%%% Calculate Inverse filter for current reconstruction
      GSB     = Gt(jx_act,:)*SB_inv;        % Ga' * Sigma_B^{-1}
      KW(:,:,j,n)   = vb_repmultiply(GSB, A_z(jx_act));
      GSB_ext = Get*SB_inv; 
      KW_ext(:,:,j,n) = vb_repmultiply(GSB_ext, A_e);
      %        SCVS(:,:,n) = SB_inv*Cov*SB_inv; 
    end
  end % Timewindow loop
else
  for j=1:Nwindow
    for n=1:Ntask
      Cov = Covs{n};     % Sg
    
      %%%% Calculate Inverse filter for current reconstruction
      A_z = a_inv(:,j,n);
		
      % Inverse filter for each session
      % SB  = Ga * a_inv *Ga' + Cov
      SB      = G * vb_repmultiply(Gt, A_z) + Cov;  % Sigma_B
      SB_inv  = inv( SB );        % Sigam_B^{-1}
      SB_cov(:,:,j,n) = SB_inv;
        
      Hj(j,n)  = - 0.5*( vb_log_det(SB) + Nch*log(sx(j,n)) );
        
      %%%% Calculate Inverse filter for current reconstruction
      GSB     = Gt(jx_act,:)*SB_inv;        % Ga' * Sigma_B^{-1}
      KW(:,:,j,n)   = vb_repmultiply(GSB, A_z(jx_act));
      %        SCVS(:,:,n) = SB_inv*Cov*SB_inv; 
    end
  end % Timewindow loop
  
  KW_ext = [];
end

%VBfilt.a  = a_inv(jx_act,:,:);
%VBfilt.sx = sx;
VBfilt.sx = sx*(Model.bsnorm_meg^2);
VBfilt.Cov = Covs;

%VBfilt.Hj = Hj;
Nmeg = size(Gact_b{1},1);
Neeg = size(Gact_e{1},1);
ix_meg = 1:Nmeg;
ix_eeg = Nmeg+1:Nmeg+Neeg;
KW(:,ix_meg) = KW(:,ix_meg)./Model.bsnorm_meg;
KW(:,ix_eeg) = KW(:,ix_eeg)./Model.bsnorm_eeg;

VBfilt.KW = KW;

if ~isempty(KW_ext)
    KW_ext(:,ix_meg) = KW_ext(:,ix_meg)./Model.bsnorm_meg;
    KW_ext(:,ix_eeg) = KW_ext(:,ix_eeg)./Model.bsnorm_eeg;    
end
VBfilt.KW_ext = KW_ext;

%VBfilt.SB_cov = SB_cov;
VBfilt.ix_act = ix_act;

VBfilt.G = Gact0{1};

VBfilt.ix_meg = ix_meg;
VBfilt.ix_eeg = ix_eeg;

%if isfield(fileinfo,'channel_id')
%	VBfilt.channel_id = fileinfo.channel_id;
%end

% Current Info
Jinfo.version   = vbmeg('version');
Jinfo.curr_type = 'Z-current';

Jinfo.Lact = Lact;
Jinfo.Wact = Wact;
Jinfo.ix_act = ix_act;
Jinfo.ix_act_ex = ix_act_ex;

% ix_act : Vertex index corresponding to active current Zact
% ix_act_ex : Vertex index corresponding to active current Jact
% Wact   : Spatial smoothing matrix of focal window
% Jact   = Wact * Zact

return

%%%% ---------------
function	[bayes_parm, ix_area] = ...
			check_arg(bayes_parm,filt_parm)

if isfield(filt_parm,'basisfile_meg')
	bayes_parm.basisfile_meg = filt_parm.basisfile_meg;
end
if isfield(filt_parm,'basisfile_eeg')
	bayes_parm.basisfile_eeg = filt_parm.basisfile_eeg;
end
if ~isfield(filt_parm,'ix_area')
	ix_area = []; 
else
	ix_area = filt_parm.ix_area; 
end
if isfield(filt_parm,'extra')
  if isfield(filt_parm.extra,'basisfile')
    bayes_parm.extra.basisfile = filt_parm.extra.basisfile;
  end
end


return
%%% --- Following fields are used for current calculation
%if isfield(filt_parm,'megfile'), 
%	bayes_parm.megfile   = filt_parm.megfile  ;
%end;
%if isfield(filt_parm,'twin_meg'), 
%	bayes_parm.twin_meg  = filt_parm.twin_meg ;
%end;
%if isfield(filt_parm,'Tperiod'), 
%	bayes_parm.Tperiod   = filt_parm.Tperiod  ;
%end;
%if isfield(filt_parm,'Tnext'), 
%	bayes_parm.Tnext     = filt_parm.Tnext    ;
%end;
%
%if ~isfield(filt_parm,'trial_average'), 
%	trial_average = ON; 
%else
%	trial_average = filt_parm.trial_average; 
%end;
%
%bayes_parm.trial_average = trial_average;
%
%if ~isfield(filt_parm,'tsubsamp'), 
%	tsubsamp = []; 
%else
%	tsubsamp = filt_parm.tsubsamp; 
%end;
%if ~isfield(filt_parm,'overlap_mode'),     
%	overlap_mode = 0; 
%else
%	overlap_mode = filt_parm.overlap_mode; 
%end;
