function vb_job_vb_meeg(varargin)
% Estimate parameters of hierarchical Bayse model using variational Bayes
% from both MEG and EEG (VBMEG public function)
% (modified of vb_job_vb.m)
%
% [syntax]
% vb_job_vb_meeg(bayes_parm)
% vb_job_vb_meeg(proj_root,bayes_parm)    [old style]
%
% [input]
% proj_root : <<string>> VBMEG project root directory. 
% bayes_parm: <<struct>> Parameters for variational Bayes method.
% --- fields of bayes_parm
%  (filename parameters)
%  brainfile       : <<string>> Cortical surface model file (.brain.mat).
%  actfile         : <<string>> Cortical activity map file (.act.mat).
%  areafile        : <<string>> Cortical area file (.area.mat).
%  megfile         : <<cell>> Cell of MEG-MAT file (.meg.mat).
%  megfile_baseline: <<cell>> Cell of MEG-MAT file (.meg.mat).
%  basisfile_meg   : <<cell>> Cell of leadfield file (.basis.mat). 
%  eegfile         : <<cell>> Cell of MEG-MAT file (.meg.mat).
%  eegfile_baseline: <<cell>> Cell of MEG-MAT file (.meg.mat).
%  basisfile_eeg   : <<cell>> Cell of leadfield file (.basis.mat). 
%
%  bayesfile       : <<string>> Current variance file (.bayes.mat). 
%
%  (noise model parameters)
%  noise_model: <<int>> Specify observation noise covariance matrix. It
%               should be noted that the covariance matrix is normalized
%               so that sum(diag(Cov)) is equal to the number of
%               sensors. In VB algorithm, a scale parameter of
%               observation noise variance (beta^{-1}) is estimated with
%               the covariance matrix (constant). 
%   noise_model=1: Identity matrix. MEG data is not used. 
%   noise_model=2: Only diagonal elements are estimated from MEG data. 
%   noise_model=3: Full covariance matrix is estimated from MEG data. 
%  noise_reg  : <<float>> Regularization parameter for observarion noise
%               covariance matrix. This value is used to suppress
%               singularity of the covariance matrix. It is effective
%               when noise_mode=2 or 3. 
%
%  (prior parameters: see Yoshioka et al., 2008, Appendix A)
%  act_key: <<string>> ID of activity map, which is used to specify prior
%           information for current variance, corresponding to
%           \hat{t}_{i}. The activity map is automatically normalized
%           (maximum=1) in estimation procedure. 
%  a0     : <<double>> 
%  a0_act : <<double>> Maximum value of current variance parameter. If
%           a0=1, this value is the same with a variance magnification
%           parameter m_{0}. 
%
%  prior_weight : Relative influence of prior information on data
%                 (from 0 to 1 real). 
%   <<The following two parameters are dependent variables of prior_weight>>
%       Ta0    : <<double>> Confidence parameter of prior current variance
%                parameters (gamma distribution) at which the activity map
%                ('act_key') takes the minimum value. 
%       Ta0_act: <<double>> Confidence parameter of prior current variance
%                parameters (gamma distribution) at which the activity map
%                ('act_key') takes the maximum value. 
%  v0     : <<double>>
%  Tv0    : <<double>>
%
%  (time window parameters)
%  twin_noise   : <<vector>> Observation noise covariance matrix is
%                 calculated using MEG data "bayes_parm.megfile_baseline"
%                 within this time window, specified by start and end. 
%  twin_baseline: <<vector>> Baseline observation noise variance
%                 (rho_{base}) and baseline current variance parameter
%                 (nu_{0,base}) are estimated from MEG data
%                 "megfile_baseline" within the time window specified by
%                 this parameter. See Yoshioka et al., 2008, Appendix A. 
%  twin_meg     : <<vector>> Current variance parameters (alpha^{-1}) are
%                 estimated from MEG data "bayes_parm.megfile" within the
%                 time window specified by this parameter. 
%  Tperiod      : <<int/vector>> Partial time window length of MEG data.
%  Tnext        : <<int/vector>> Moving step length of partial time
%                 window of MEG data. 
%
%  (forward model parameters: see Yoshioka et al., 2008, Appendix B)
%  forward_model  : <<string>> Set to 'focal'. 
%  area_key       : <<string>> ID of cortical area at which current
%                   dipoles are assumed. 
%  reduce         : <<float>> Current dipole reduction ratio (0-1). 
%  Rfilt          : <<float>> Smoothing filter radius in [m]. Elements of
%                   matrix W is determined from this parameter and
%                   inter-dipole distance. 
% .area_key_global: <<string>> ID of cortical area at whith current
%                   dipoles are assumed in estimation of baseline
%                   variance parameter (nu_{0,base}). 
% .reduce_global  : <<float>> Current dipole reduction ratio in
%                   estimation of baseline variance parameter
%                   (nu_{0,base}) (0-1). 
% .Rfilt_global   : <<float>> Smoothing filter radius in [m] in
%                   estimation of baseline variance parameter
%                   (nu_{0,base}). 
% .patch_norm     : <<bool>> If true, patch area normalization is
%                   applied. That is, dipole current moment (Am) is
%                   replaced by its (area) density (Am/m^2). 
% .expand_spatial_filter: <<bool>> Set to 'true'. 
% .spatial_smoother : <<string>> smoothing leadfield  
%                      = 'std' : using Standard brain(nextIX, nextDD, xx)
%                      = 'subj': using Individual brain(subj.nextIX,
%                                subj.nextDD, subj.xx)
% .remove_crossed_area  : <<bool>> If true, cortical area specified by
%                         'area_key' is removed in estimation of baseline
%                         variance parameter (nu_{0,base}). 
%
%  (extra-dipole parameters)
%  extra            : <optional> <<struct>> Parameter set of
%                     extra-dipole. 
%  extra.basisfile  : <<string>> Filename of leadfield file (.basis.mat)
%                     for extra-dipole. 
%  extra.a0         : <<vector>> Alpha_{0}^{-1} for all extra-dipoles. 
%  extra.Ta0        : <<vector>> Gamma_{0} for all extra-dipoles. 
%  extra.a0_physical: <<bool>> If true, a0 is interpreted as physical
%                     unit (Am^2, current variance). Specifically, 
%                     vb_parm.a0 is set to extra.a0/sx0, where sx0 is the
%                     baseline of observation noise variance. 
%
%  (algorithm parameters)
%  Ntrain    : <<int>> Iteration number of VB algorithm. 
%  Npre_train: <<int>> Iteration number of fast VB algorithm. 
%  skip      : <<int>> Calculate free energy at each 'skip' steps. 
%  update_sx : <<bool>> If true, observarion noise variance is
%              simultaneously updated with prior current variance
%              parameter (alpha_{0}). 
%  update_v  :   (bool)
%  Fdmin     : <<float>> Constant for convergence test of VB algorithm. 
%  a_min     : <<float>> Minimum value of alpha_{0}^{-1} for stability of
%              the VB alrogithm. 
%  a_max     : <<float>> Maximum value of alpha_{0}^{-1} for stability of
%              the VB alrogithm. 
%  cont_pr   : <<bool>> If true, estimated current variance parameters
%              (alpha^{-1}) are used as the initial value of current
%              variance parameters in the next time window. 
%
%  (Parameters for soft normal constraints (not publicly supported in
%  current version of VBMEG))
%  soft_mode           : <<bool>>
%  variance_orientation: <<bool>>
%  var_max             : <<float>>
%  tan_var             : <<float>>
%  cosval              : <<float>>
%
%  (preprocessing parameters (not recommended to use))
%  .trial_average  : <<bool>> If true, trial average is applied to the
%                    original MEG data. 
%  .temporal_filter: <<bool>> If true, temporal smoothing filter is
%                    applied to the original MEG data. 
%
% [output]
% Estimation result will be saved to the file specified by
% bayes_parm.bayesfile with the following variables: 
%  Model.a   : <<vector>> (alpha_{i}^{-1})
%  Model.sx  : <<float>> (beta^{-1})
%  Model.v   : <<float>> (nu_{0,base})
%  Model.a0  : <<vector>> (alpha_{0i}^{-1}).
%  Model.sx0 : <<float>> (rho_{base}=beta_{base}^{-1})
%  Model.v0  : <<float?>> (nu_{0,base})
%  Model.ix  : <<vector>> Dipole indices of Model.a. 
%  Model.Cov : <<cell?>> Noise covariance matrix.
%  bayes_parm: <<struct>> All estimation parameter. 
%  vb_parm   : <<struct>> Parameters for VB algorithm. 
%  Info      : <<struct>> Free energy, etc.
%  Pointlist : <<struct>> Extra dipole information.
%
%  [note]
%  1. 'bayes_parm.reduce', J-current and Z-current
%  Cortical current J is represented by an auxillary parameter Z using
%  equation J=WZ, where W is Gaussian smoothing matrix (see Yoshioka et
%  al., 2008, Appendix A). The number of current dipoles I (=size(J,1))
%  is reduced to ('bayes_parm.reduce' times I), which is the same with
%  the number of current variance parameters (size(Model.a,1)). J and Z
%  are referred to as J-current and Z-current, respectively. 
%
%  2. Time window parameters
%  Firstly, it is noted that the unit of time window parameters (e.g.,
%  'twin_meg') is the sampling point depending on the sampling frequency
%  of M/EEG measurement. In addition, these values are NOT relative to
%  the pre-trigger, but relative to the start of each trial. For
%  simplicity, we assume 1 ms sampling data in the following
%  explanation. 
%  MEG data is separated into a number of time windows and model
%  parameters are estimated for each time window. The following setting
%  is a simple example: 
%   >> bayes_parm.twin_meg = [501 1000];
%   >> bayes_parm.Tperiod  = 500;
%   >> bayes_parm.Tnext    = 500;
%  In this case, model parameters are estimated from (500 x N) samples
%  501 ms after each trial, where N denotes the number of trials. As a
%  result, size(Model.a) becomes [I 1], where I denotes the number of
%  current dipoles of Z-current. In the next example, whole time window
%  is separated into 5 time windows: 
%   >> bayes_parm.twin_meg = [501 1000];
%   >> bayes_parm.Tperiod  = [100 100 100 100 100];
%   >> bayes_parm.Tnext    = [1   101 201 301 401];
%  The start of time windows are 1, 101, ..., 401 ms and the length is
%  100 ms. The following parameter set is the same with the above
%  example: 
%   >> bayes_parm.twin_meg = [501 1000];
%   >> bayes_parm.Tperiod  = 100; 
%   >> bayes_parm.Tnext    = 100;
%  'Tperiod' and 'Tnext' are interpreted as the length and moving steps
%  of the time window, respectively. Time windows can be overlapped: 
%   >> bayes_parm.twin_meg = [501 1000];
%   >> bayes_parm.Tperiod  = [200 200 200 200];
%   >> bayes_parm.Tnext    = [1   101 201 301];
%  Within the overlapped periods, dipole currents are calculated by the
%  average of dipole currents using estimated parameters corresponding to
%  overlapped time windows. This is implemented not in job_vb.m, but in
%  current estimation functions. 
%  Different length of time windows can be specified: 
%   >> bayes_parm.twin_meg = [401 1000];
%   >> bayes_parm.Tperiod  = [100 500];
%   >> bayes_parm.Tnext    = [1 101];
%  In this case, model parameters are estimated separately for
%  pre-trigger (100 ms) and post-trigger (500 ms) of MEG data. 
%
% [reference]
% Sato et al. (2004) NeuroImage 23, 806-826. 
% Yoshioka et al. (2008) NeuroImage 42, 1397-1413. 
%
% [history]
% 2017-06-15 Yusuke Fujiwara refined
% 2017-03-16 rhayashi
%   spatial_smoother is added.
% 2017-03-10 Yusuke Fujiwara
%   MEG+EEG
% 2008-06-27 Taku Yoshioka
% 2008-07-07 Masa-aki Sato
%   Model.Cov is added
% 2008-08-08 Taku Yoshioka
%   Support for physical unit for extra dipole.
% 2008-10-06 Taku Yoshioka
%   Bug fix (physical unit for extra dipole)
% 2008-12-03 Taku Yoshioka
%   Use vb_disp() for displaying messages
% 2017-11-14 oyamashi
%   prior_weight implementation.
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if length(varargin) == 1
  proj_root = [];
  bayes_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  bayes_parm = varargin{2};
end

proj_root = vb_rm_trailing_slash(proj_root);

const = vb_define_verbose;
VERBOSE_LEVEL_INFO = const.VERBOSE_LEVEL_INFO;

%
% Change path to absolute path
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('----- New VBMEG -----',VERBOSE_LEVEL_INFO)
if ~isfield(bayes_parm, 'spatial_smoother')
    dp = vb_set_bayes_default_parameters;
    bayes_parm.spatial_smoother = dp.spatial_smoother;
end
bayes_parm_abs = vb_parm_absolute_path(proj_root, bayes_parm);

Nsession = length(bayes_parm_abs.eegfile_baseline);
for nf = 1:Nsession
    bayes_parm_abs.eegfile_baseline{nf} = fullfile(proj_root, bayes_parm_abs.eegfile_baseline{nf});
    bayes_parm_abs.eegfile{nf} = fullfile(proj_root, bayes_parm_abs.eegfile{nf});
end
bayes_parm_abs.basisfile_meg = fullfile(proj_root, bayes_parm_abs.basisfile_meg);
bayes_parm_abs.basisfile_eeg = fullfile(proj_root, bayes_parm_abs.basisfile_eeg);


%
% Preparation of lead fields
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('----- Leadfield matrix preparation',VERBOSE_LEVEL_INFO); 

lf_parm.brainfile = bayes_parm_abs.brainfile;
lf_parm.areafile = bayes_parm_abs.areafile;
lf_parm.patch_norm = bayes_parm_abs.patch_norm;
lf_parm.expand_spatial_filter = bayes_parm_abs.expand_spatial_filter;
lf_parm.spatial_smoother = bayes_parm_abs.spatial_smoother;

% % Global window
% vb_disp('--- Lead field matrix of global window ',VERBOSE_LEVEL_INFO);
% lf_parm.basisfile = bayes_parm_abs.basisfile_global;
% lf_parm.area_key = bayes_parm_abs.area_key_global;
% lf_parm.reduce = bayes_parm_abs.reduce_global;
% lf_parm.Rfilt = bayes_parm_abs.Rfilt_global;
% if strcmp(bayes_parm_abs.forward_model,'focal+global')==1 ...
%       && bayes_parm_abs.remove_crossed_area==ON,
%   % Remove focal area from global area
%   lf_parm.remove_area_key = bayes_parm_abs.area_key;
% else 
%   lf_parm.remove_area_key = []; 
% end
% [Gall_meg, ix_all] = vb_leadfield_preparation(lf_parm);


%%%%%%%%%%%%%%%%%%% MEG %%%%%%%%%%%%%%%%%%%%%
% Focal window
vb_disp('--- Lead field matrix of focal window ',VERBOSE_LEVEL_INFO);
lf_parm.basisfile = bayes_parm_abs.basisfile_meg;
lf_parm.area_key = bayes_parm_abs.area_key;
lf_parm.reduce = bayes_parm_abs.reduce;
lf_parm.Rfilt = bayes_parm_abs.Rfilt;
lf_parm.remove_area_key = [];
[Gact_meg, ix_act_meg] = vb_leadfield_preparation(lf_parm);


%%%%%%%%%%%%%%%%%%% EEG %%%%%%%%%%%%%%%%%%%%%
lf_parm.basisfile = bayes_parm_abs.basisfile_eeg;
[Gact_eeg, ix_act_eeg] = vb_leadfield_preparation(lf_parm);

ix_act = ix_act_meg;


%
% Normalization of lead fields
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Normalization constant of leadfield
Nmeg = size(Gact_meg{1},1);
Neeg = size(Gact_eeg{1},1);

for n = 1: Nsession
    GGall_meg = sum(Gact_meg{n}.^2, 2);
    Bnorm_meg(n)  = sum(GGall_meg)/length(GGall_meg);
    GGall_eeg = sum(Gact_eeg{n}.^2, 2);
    Bnorm_eeg(n)  = sum(GGall_eeg)/length(GGall_eeg);
end
bsnorm_meg = sqrt(mean(Bnorm_meg));
bsnorm_eeg = sqrt(mean(Bnorm_eeg));

% Normalization of leadfield
for n=1:Nsession
  nGact_meg{n} = Gact_meg{n}/bsnorm_meg;
  nGact_eeg{n} = Gact_eeg{n}/bsnorm_eeg;
  nGact{n} = cat(1,nGact_meg{n},nGact_eeg{n});
end
nGall_meg = nGact_meg;
nGall_eeg = nGact_eeg;
nGall = nGact;


%
% Observation noise estimation
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('----- Observation noise model specification', ...
        VERBOSE_LEVEL_INFO); 

%Cov Normalized noise covariance matrix

%MEG covariance
tmp_parm = bayes_parm_abs;
%tmp_parm.basisfile = tmp_parm.basisfile_meg;
[Cov_meg,sx0_meg] = vb_observation_noise_specification(tmp_parm);

%EEG covariance
%tmp_parm.basisfile = tmp_parm.basisfile_eeg;
tmp_parm.megfile_baseline = tmp_parm.eegfile_baseline;
[Cov_eeg,sx0_eeg] = vb_observation_noise_specification(tmp_parm);

sx0_meg2 = sx0_meg/(bsnorm_meg.^2);
sx0_eeg2 = sx0_eeg/(bsnorm_eeg.^2);

%ix_meg = 1:size(Cov_meg{1},1);
%ix_eeg = ix_meg(end)+1:ix_meg(end)+size(Cov_eeg{1},1);

for nf = 1:Nsession
    Cov{nf} = blkdiag(Cov_meg{nf},(sx0_eeg2/sx0_meg2).*Cov_eeg{nf});
end

sx0 = sx0_meg2;

%
% MEG data preparation
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[B0, Ntrial, tmp, Tsample] = vb_megdata_preparation(bayes_parm_abs);

% EEG data
megfile = bayes_parm_abs.megfile;
bayes_parm_abs.megfile = bayes_parm_abs.eegfile;
[E0] = vb_megdata_preparation(bayes_parm_abs);

for n=1:Nsession
    B0{n} = B0{n}/bsnorm_meg;
    E0{n}  = E0{n}/bsnorm_eeg;
    B{n}  = cat(1,B0{n},E0{n});
end
bayes_parm_abs.megfile = megfile;

%
% Check variable consistency
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_check_variable(B, Cov, nGact, nGall);

%
% Estimation of baseline current variance
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('----- Baseline current variance estimation',VERBOSE_LEVEL_INFO);
      
% current baseline estimation at pretrigger period
% together with spherical sensor noise estimation
%MEG
tmp_parm = bayes_parm_abs;
tmp_parm.megfile_baseline = tmp_parm.megfile_baseline;
tmp_parm.basisfile{1} = tmp_parm.basisfile_meg;
tmp_parm.basisfile_global{1} = tmp_parm.basisfile_meg;
[v0m, sx0b] = vb_baseline_variance_acquisition(tmp_parm, nGall_meg);

%EEG
tmp_parm.megfile_baseline = tmp_parm.eegfile_baseline;
tmp_parm.basisfile{1} = tmp_parm.basisfile_eeg;
tmp_parm.basisfile_global{1} = tmp_parm.basisfile_eeg;
[v0e, sx0e] = vb_baseline_variance_acquisition(tmp_parm, nGall_eeg);

v0 = (v0m + v0e)/2;

%
% Set VB parameters
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Njact = size(nGact{1},2);
Njall = size(nGall{1},2);
%vb_parm = vb_set_vb_parm(bayes_parm_abs, B, Njact, Njall, v0, sx0);
vb_parm = vb_set_vb_parm(tmp_parm, B, Njact, Njall, v0, sx0);

%
% Set prior parameters
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
prior_parm.ix = ix_act;
prior_parm.v0 = v0;
prior_parm.a0 = bayes_parm.a0;
prior_parm.a0_act = bayes_parm.a0_act;

w = bayes_parm.prior_weight;
prior_parm.Ta0 = Ntrial * Tsample * w / (2*(1-w));
prior_parm.Ta0_act = prior_parm.Ta0;

prior_parm.act = vb_get_act(bayes_parm_abs.actfile, ...
                         bayes_parm_abs.act_key);
[a0,Ta0] = vb_set_vb_prior(prior_parm);
vb_parm.a0 = a0;
vb_parm.Ta0 = Ta0;

%
% Modify prior parameters for soft normal constraint
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if bayes_parm.variance_orientation == ON, 
  soft_parm.ix = ix_act;
  soft_parm.a0 = a0;
  soft_parm.Ta0 = Ta0;
  soft_parm.Norient = vb_parm.Norient;
  soft_parm.soft_mode = vb_parm.soft_mode;
  soft_parm.brainfile = bayes_parm_abs.brainfile;
  soft_parm.var_max = vb_parm.var_max;
  soft_parm.tan_var = vb_parm.tan_var;
  [a0,Ta0] = set_soft_prior(soft_parm);
  vb_parm.a0 = a0;
  vb_parm.Ta0 = Ta0;
end

%
% Add extra dipole
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if isfield(bayes_parm_abs,'extra') & ~isempty(bayes_parm_abs.extra), 
  vb_struct2vars(bayes_parm_abs,{'extra'});

  % prior for extra dipole
  if extra.a0_physical == true, e0 = extra.a0/sx0.*bsnorm^2;
  else e0 = extra.a0*v0; end
  vb_parm.a0 = [vb_parm.a0; e0];
  vb_parm.Ta0 = [vb_parm.Ta0; extra.Ta0];
  
  for i=1:length(nGact)
    extra_basis = vb_load_basis(extra.basisfile{i});
    nGact{i} = [nGact{i} (extra_basis./bsnorm)'];
  end
  load(extra.basisfile{1},'Pointlist');
  Nextra = length(Pointlist);  
  vb_parm.Njact = vb_parm.Njact+Nextra*3;
  vb_parm.Nvact = vb_parm.Nvact+Nextra*3;
else
  Pointlist = [];
  Nextra = 0;
end

%
% Wiener filter (removed from the previous version)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Model = [];

%
% Estimation with focal (and global) window
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

switch bayes_parm_abs.forward_model
 case 'focal+global',
  [Model,Info] = vbmeg_ard_estimate2(B, nGall, nGact, ...
                                     Cov, vb_parm, Model);
 case 'focal', 
  vb_parm.v0  = 0;
  vb_parm.Tv0 = 0; 
  [Model,Info] = vbmeg_ard_estimate4(B, [], nGact, ...
                                     Cov, vb_parm, Model);

otherwise,
  error(['---- no such forward model : ' ...
         bayes_parm_abs.forward_model '----']);
end

Model.sx0 = sx0;

%
% Denormalization
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if isfield(bayes_parm,'DEBUG') && bayes_parm.DEBUG==1
  % For DEBUG
  fprintf('Old saving mode\n')
else
  %  Alpha is scaled back by bsnorm
  %fprintf('Alpha is scaled back by bsnorm\n')
  %Model.a = (Model.a)./bsnorm^2;
  %Model.v = (Model.v)./bsnorm^2;
  %Model.a0 = (vb_parm.a0)./bsnorm^2;
  %Model.v0 = (vb_parm.v0)./bsnorm^2;
  Model.Cov = Cov;
  Model.ix = ix_act;
  Model.bsnorm_meg = bsnorm_meg;
  Model.bsnorm_eeg = bsnorm_eeg;
end

%
% Split variance parameters into cortial and extra dipoles
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if Nextra>0, % Nextra is the number of extra points
  Model_ext.sx = Model.sx;
  Model_ext.sx0 = Model.sx0;
  Model_ext.a = Model.a(end-Nextra*3+1:end,:);
  Model_ext.a0 = Model.a0(end-Nextra*3+1:end,:);
  Model.a = Model.a(1:end-Nextra*3,:);
  Model.a0 = Model.a0(1:end-Nextra*3,:);
else
  Model_ext = [];
end


%
% Save result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fprintf('----- Save estimation result in %s\n',bayes_parm_abs.bayesfile)

%bayes_parm = bayes_parm_abs;
vb_fsave([bayes_parm_abs.bayesfile], ...
        'Model','Model_ext','bayes_parm','vb_parm',...
         'Info','Pointlist');

% project_file save
proj_file = get_project_filename;
if isempty(proj_file)
    return;
end

project_file_mgr('load', proj_file);
project_file_mgr('add', 'bayes_parm', bayes_parm);
